/******************************************************************************
 * (C) Copyright 2000 by Agilent Technologies GmbH. All rights reserved.      *
 ******************************************************************************/

/***********************************************************************
 * File name     : board.c
 *                 board and status properties and generic functions
 ***********************************************************************/
#include <math.h>

#include <xtypedef.h>

#include <xpciapi.h>

#include <xaddrmap.h>
#include <xboard.h>
#include <xdynamic.h>
#include <xerrcapi.h>
#include <xiocommo.h>
#include <xregcons.h>
#include <xregx10.h>
#include <xregx11.h>
#include <xsession.h>
#include <xctdefs.h>

#include <xdownld.h>

extern bx_errtype BestXTraceStatusRead (bx_handletype handle,
                    bx_statustype prop,
                    bx_int32ptr pValue);

/* Pointer to board properties in database */
#define BOARD(prop) (bx_handlearray[handle].db->BoardProp[prop])

/********************************************************************
  Local function prototypes *****************************************
 *******************************************************************/

static bx_errtype BestXPCIFrequencyRead(bx_handletype,bx_int32,bx_int32,bx_int32ptr );
static bx_errtype EnableInterrupts(bx_handletype);
static bx_errtype BestXStatusTestRead(bx_handletype,bx_int32 *);
static bx_errtype BestXStatusCfgRead(bx_handletype,bx_int32 *);
static bx_errtype AbortClear(bx_handletype);
static bx_errtype BestXWaitForDiagXfer(bx_handletype);

/********************************************************************
  Best Status *******************************************************
 *******************************************************************/

/*---------------------------------------------------------------------------*
 * BestXStatusClear
 *
 * Purpose: This function clears different card stati
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXStatusClear(
  bx_handletype handle,
  bx_statustype prop     /* What to clear */
)
{
  bx_int32 val;

  BX_DECLARE_FUNCNAME ("BestXStatusClear [statusclear]");
  
  BX_TRY_VARS_NO_PROG;

  /*
  BX_TRY_FCT_PARAM(2,prop!=BX_STAT_OBS_ERR && prop!=BX_STAT_SPLITFAIL);
  */

  BX_TRY_BEGIN 
  {
    switch (prop)
    {
      case BX_STAT_OBS_ERR:
        /* Clear all observer errors */
        if (BestXHasMephisto(handle))
        {
          BX_TRY(BestXDirectRegWrite(handle,BX_REG_ERR_REG_RST_STRB_M,sizeof(bx_int16),0));
        }
        else
        {
          BX_TRY(BestXDirectRegWrite(handle,BX_REG_OBS_CTRL_STATUS_REG_F,sizeof(bx_int16),1<<3));
        }
        break;

      case BX_STAT_SPLITFAIL:
        /* Clear split error message */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DEC_SPLIT_ERR_MSG_REG,sizeof(bx_int16),0x0));
        break;

      case BX_STAT_LASTRESET:
        /* Clear bit 6 by writing a one */
        if (BestXHasMephisto(handle))
        {
          /* Caution:
             Bit 6 (PCIRST_occ) is already used by FW to switch card into runmode. */
          BX_TRY(BestXDirectRegWrite(handle,BX_REG_SYS_STAT_MONITOR,sizeof(bx_int32),1<<6));
        }
        else
        {
          /* Bit 6 (PCIRST_occ) already used by FW, so we use bit 16 (PCIRST_occ_2) here */
          BX_TRY(BestXDirectRegWrite(handle,BX_REG_SYS_STAT_MONITOR,sizeof(bx_int32),1<<16));
        }
        break;

      case BX_STAT_CMP_FAIL:
        /* Clear data compare flag */
        /* This covers all the BX_STAT_CMP_XXX items, which are only valid 
           after a data miscompare has happened. 
        */

        BX_TRY(BestXDirectRegWrite(handle,BX_REG_ERR_CLEAR_STRB,sizeof(bx_int16),0x0));
        if (BestXHasFaust(handle))
        {
          /* Clear datamem error */
          BX_TRY(BestXDirectRegWrite(handle,BX_REG_ERR_CLEAR_STRB_F,sizeof(bx_int16),0x0));
        }
        break;
        
      case BX_STAT_ERR_WRPAR:
      case BX_STAT_ERR_WRPAR64:
        /* These cannot be cleared individually ! */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_STAT_PAR_ERR_REG,sizeof(bx_int16),0x0));
        break;

      case BX_STAT_IABORT:
      case BX_STAT_TABORT:
        /* These cannot be cleared individually ! */
        BX_TRY(AbortClear(handle));
        break;

      case BX_STAT_ERR_SERR: /* Direct access to status register of config space here */
        BX_TRY(BestXDirectRegRead(handle,BX_REG_DEV_VEN_ID_REG+4,sizeof(bx_int32),&val));
        val &= ~(1<<30); /* Clear bit 30 */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DEV_VEN_ID_REG+4,sizeof(bx_int32),val));
        break;

      case BX_STAT_ERR_PERR: /* Direct access to status register of config space here */
        BX_TRY(BestXDirectRegRead(handle,BX_REG_DEV_VEN_ID_REG+4,sizeof(bx_int32),&val));
        val &= ~(1<<31); /* Clear bit 31 */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DEV_VEN_ID_REG+4,sizeof(bx_int32),val));
        break;
  
      case BX_STAT_INTA: 
       BX_TRY (BestXInterruptClear (handle, 1UL));
        break;

     case BX_STAT_INTB: 
       BX_TRY (BestXInterruptClear (handle, 2UL));
        break;

     case BX_STAT_INTC: 
       BX_TRY (BestXInterruptClear (handle, 4UL));
        break;

     case BX_STAT_INTD: 
       BX_TRY (BestXInterruptClear (handle, 8UL));
        break;

     default:
        BX_E_ERROR_MSG_SET("BestXStatusClear: This status cannot be cleared");
        BX_TRY_ERROR(BX_E_ERROR);
        break;
    }
  }

  BX_ERRETURN (BX_TRY_RET);
}


/*---------------------------------------------------------------------------*
 * BestXStatusRead
 *
 * Purpose: Reads info about the current card's status
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXStatusRead(
    bx_handletype handle,
    bx_statustype prop,
    bx_int32 *val
)
{
  bx_int32 readval = 0;
  bx_int32 t=0;

  BX_DECLARE_FUNCNAME("BestXStatusRead [statusread]");
  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_TRY(BestXParamProbe(handle, BX_PARAM_STATUS, (bx_int32)prop));
    BX_FCT_PARAM_NULL_POINTER_CHK(val);

    switch (prop)
    {
      case BX_STAT_TABORT:
        BX_TRY(BestXDirectRegRead(handle,BX_REG_MSM_STAT_MON,sizeof(bx_int16),&readval));
        readval = (readval & 1<<0 ? 1:0); /* Extract bit 0 */
        break;
      case BX_STAT_IABORT:
        BX_TRY(BestXDirectRegRead(handle,BX_REG_MSM_STAT_MON,sizeof(bx_int16),&readval));
        readval = (readval & 1<<1 ? 1:0); /* Extract bit 1 */
        break;
      case BX_STAT_ERR_PERR: /* Direct access to status register of config space here */
        BX_TRY(BestXDirectRegRead(handle,BX_REG_DEV_VEN_ID_REG+4,sizeof(bx_int32),&readval));
        readval = readval>>31 & 0x1; /* Extract bit 31 */
        break;
      case BX_STAT_ERR_SERR: /* Direct access to status register of config space here */
        BX_TRY(BestXDirectRegRead(handle,BX_REG_DEV_VEN_ID_REG+4,sizeof(bx_int32),&readval));
        readval = readval>>30 & 0x1; /* Extract bit 30 */
        break;
      case BX_STAT_ERR_WRPAR:
        BX_TRY(BestXDirectRegRead(handle,BX_REG_STAT_PAR_ERR_REG,sizeof(bx_int16),&readval));
        readval = readval>>1 & 0x1; /* Extract bit 1 */
        break;
      case BX_STAT_ERR_WRPAR64:
        BX_TRY(BestXDirectRegRead(handle,BX_REG_STAT_PAR_ERR_REG,sizeof(bx_int16),&readval));
        readval = readval & 0x1; /* Extract bit 0 */
        break;
      case BX_STAT_INTA:
        BX_TRY (BestXInterruptStatus(handle, &readval));
        readval = (readval & 0x1) ? 1UL : 0UL; /* Extract bit 0 */
        break;
      case BX_STAT_INTB:
        BX_TRY (BestXInterruptStatus(handle, &readval));
        readval = (readval & 0x2) ? 1UL : 0UL; /* Extract bit 1 */
        break;
      case BX_STAT_INTC:
        BX_TRY (BestXInterruptStatus(handle, &readval));
        readval = (readval & 0x4) ? 1UL : 0UL; /* Extract bit 2 */
        break;
      case BX_STAT_INTD:
        BX_TRY (BestXInterruptStatus(handle, &readval));
        readval = (readval & 0x8) ? 1UL : 0UL; /* Extract bit 3 */
        break;

     case BX_STAT_OBS_ERR:
        /* We do not need to write to the shadow strobe register here ! */
        if (BestXHasMephisto(handle))
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_STAT_OBS_ERR_REG_M,sizeof(bx_int16),&readval));
          readval = readval & 0x1; /* Extract bit 0 */
        }
        else
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_OBS_CTRL_STATUS_REG_F,sizeof(bx_int16),&readval));
          readval = readval & 0x1; /* Extract bit 0 */
        }
        break;

      case BX_STAT_TRC_RUNNING:
        BX_TRY(BestXDirectRegRead(handle,BX_REG_TR_STATUS_REG,sizeof(bx_int32),&readval));
        readval = readval>>2 & 0x1; /* Extract bit 2 */
        break;
      case BX_STAT_TRC_TRIGGER:
        BX_TRY(BestXDirectRegRead(handle,BX_REG_TR_STATUS_REG,sizeof(bx_int32),&readval));
        readval = readval & 0x1; /* Extract bit 0 */
        break;
      case BX_STAT_TRC_MEMFULL:
      case BX_STAT_TRC_TRIGPOS:
      case BX_STAT_TRC_LINES:
        BX_TRY(BestXTraceStatusRead(handle, prop, &readval)); /* see xladata.c */
        break;
      case BX_STAT_PCIXPOWER:
        if (BestXIsOffline(handle))
        {
          readval=1; /* Power always good ! */
        }
        else
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_SYS_STAT_MONITOR,sizeof(bx_int32),&readval));
          readval = readval>>3 & 0x1; /* Extract bit 3 */
        }
        break;
      case BX_STAT_PCIXMODE:
        BX_TRY(BestXDirectRegRead(handle,BX_REG_PCIX_MEN_REG,sizeof(bx_int16),&readval));
        readval = readval>>2 & 0x1; /* Extract bit 2 */
        break;
      case BX_STAT_PCIRESET:
        if (BestXIsOffline(handle))
        {
          readval=1; /* RST# always high (not asserted) */
        }
        else
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_SYS_STAT_MONITOR,sizeof(bx_int32),&readval));
          readval = readval>>5 & 0x1; /* Extract bit 5 */
          /* Caution: RST# is low active, so returning a zero here means RST# asserted */
        }
        break;
      case BX_STAT_RESETCODE:
        /* Caution:
           The order of these bits is not like in spec (and in beckplane)
           Here/Faust:     PERR#,TRDY#,STOP#,DEVSEL# (bits3-0)
           Spec/beckplane: PERR#,DEVSEL#,STOP#,TRDY#
           (=>caller may want to exchange bits 0 and 2)
        */
        if (BestXHasMephisto(handle))
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_SYS_STAT_MONITOR,sizeof(bx_int32),&readval));
          readval = readval & 0x7; /* Extract bits 0-2 */
          /* Mode 2 not available -> Set bit 3 (to indicate PERR# is deasserted) */
          readval |= 1<<3;
        }
        else
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_SYS_RESETPATT_F,sizeof(bx_int16),&readval));
          readval = readval & 0xf; /* Extract bits 0-3 */
        }
        break;
      case BX_STAT_LASTRESET:
        BX_TRY(BestXDirectRegRead(handle,BX_REG_SYS_STAT_MONITOR,sizeof(bx_int32),&readval));
        if (BestXHasMephisto(handle))
        {
          /* Caution:
             Bit 6 (PCIRST_occ) is already used by FW to switch card into runmode.
             So we will nearly always return a zero here ! */
          readval = readval>>6 & 0x1; /* Extract bit 6 */
        }
        else
        {
          /* Bit 6 (PCIRST_occ) already used by FW, so we use bit 16 (PCIRST_occ_2) here */
          readval = readval>>16 & 0x1; /* Extract bit 16 */
        }
        break;
      case BX_STAT_BUSSPEED:
        BX_TRY(BestXPCIFrequencyRead(handle,0x10000,0,&readval));
        break;
      case BX_STAT_PLLSPEED:
        BX_TRY(BestXPCIFrequencyRead(handle,0x10000,1,&readval));
        break;
      case BX_STAT_BUSWIDTH:
        /* Check, wether internal logic sees a 32 or 64 bit bus.
           This value is the same as seen during PCIRST at the REQ64 signal,
           if motherboard is working correct. 
           If not, the (internal) bus width can 
           be overridden by bit 3 ('1'=32 bit, '0'=64 bit) */
        BX_TRY(BestXDirectRegRead(handle,BX_REG_PCI_BW_CNTRL_REG,sizeof(bx_int16),&readval));
        readval = readval>>4 & 0x1; /* Extract bit 4 */
        readval = (readval?64:32);
        break;
      case BX_STAT_PIGGYBACKID:
        {
          bx_int32 hasPiggy; /* Wether there is an option board at all */
          bx_int32 piggyID;     /* Which option board is hasPiggy      */

          if (BestXIsOffline(handle))
          {
            readval = (BestXIsDeep(handle) ? 1UL : 0UL);
            break;
          }

          BX_TRY(BestXDirectRegRead(handle,BX_REG_XILINX_PCI,2UL,&hasPiggy));
          hasPiggy &= BX_MASK_XILINX_PCI_PIGGYBACK; /* 0=something's onboard */
          hasPiggy = !hasPiggy; /* 0=nothing's onboard */
          
          if (hasPiggy)
          {
            /* 30er will never get here */

            /* Some option board is onboard;     
               now find out which one */
            BX_TRY(BestXDirectRegRead(handle,BX_REG_PIGGY_ID, 2UL,&piggyID));
      
            piggyID &= BX_PIGGY_ID_MASK; /* only lower 3 bits are valid */
            if (piggyID==BX_PIGGY_ID_DT)
            {
              /* We've got a deep trace board onboard */
              readval=1UL;
            }
            else if (piggyID==BX_PIGGY_ID_FP)
            {
              /* option 120 board onboard */
              readval=2UL;
            }    
            else
            {
              BX_E_ERROR_MSG_SET("BestXStatusRead: Fatal error: Unknown option board found");
              BX_TRY_FAIL(BX_E_ERROR);
            }
          }
          else
          { 
            /* no piggy back onboard at all */
            /* 30er will go here */

            readval=0;
          }
        }
        break;
      case BX_STAT_TRIGIO:
        BX_TRY(BestXDirectRegRead(handle,BX_REG_STAT_IAE_REG,sizeof(bx_int16),&readval));
        readval = readval & 0x7; /* Extract bits 0-3 */
        break;
      case BX_STAT_INVISIBLE:
        /* DIP S1 of S101 (S102 30er) */
        BX_TRY(BestXDirectRegRead(handle,BX_REG_GEN_DEC_PROP_REG,sizeof(bx_int16),&readval));
        readval = (readval&2 ? 1:0); /* Extract bit 1 */
        break;
      case BX_STAT_USRSETT: /* Powerup loader */
        /* DIP S2 of S101 (S102 30er) */
        BX_TRY(BestXDirectRegRead(handle,BX_REG_DBI_STATUS_REG,sizeof(bx_int16),&readval));
        readval = (readval&1<<15 ? 1:0); /* Extract bit 15 */
        break;
      case BX_STAT_STDRECOVR: /* Stay in Core */
        /* DIP S3 of S101 (S102 30er) */
        BX_TRY(BestXDirectRegRead(handle,0xfff906,sizeof(bx_int16),&readval));
        readval = (readval&1<<7 ? 1:0); /* Extract bit 7 */
        break;
      case BX_STAT_TEST:
        BX_TRY(BestXStatusTestRead(handle,&readval));
        break;
      case BX_STAT_CFGSTAT:
        BX_TRY(BestXStatusCfgRead(handle,&readval));
        break;
      case BX_STAT_CLK:
        BX_TRY(BestXDirectRegRead(handle,BX_REG_SYS_CLK_MODE_REG,sizeof(bx_int16),&readval));
        if (BestXHasMephisto(handle))
        {
          readval = readval>>15  & 0x1; /* Extract bit 15 */
        }
        else
        {
          readval = readval>>9  & 0x1; /* Extract bit 9 */
        }
        readval = (readval?0 : 1 /* PLL off */);
        break;
      case BX_STAT_SPLITFAIL:
        BX_TRY(BestXDirectRegRead(handle,BX_REG_DEC_SPLIT_ERR_MSG_REG,sizeof(bx_int16),&readval));
        readval = readval & 1; /* Extract bit 0 */
        break;

      /***********************************************************************
       *********************  COMPARE ERROR  *********************************
       **********************************************************************/

      case BX_STAT_CMP_FAIL:
        BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_STATUS_M,sizeof(bx_int16),&readval));
        if (BestXHasMephisto(handle))
        {
          readval &= 1; /* Extract bit 0 */
        }
        else
        {
          /* Bit 0: DataGenerator compare error occurred.
             Bit 8: DataMemory compare error occurred.
             HW guarantees that not both bits are set at the same time.
          */
          readval=(readval&(1<<0)|readval&(1<<8)?1:0);
        }
        break;
      case BX_STAT_CMP_CMD:
        if (BestXHasMephisto(handle))
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_CMDBE_M,sizeof(bx_int32),&readval));
        }
        else
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_STATUS_M,sizeof(bx_int16),&readval));
          if (readval & 1<<0)
          {
            /* Compare error from datagenerator (only)*/
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_CMDBE_M,sizeof(bx_int32),&readval));
          }
          else
          {
            /* Compare error from datamemory (or no error) */
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_CMDBE_F,sizeof(bx_int32),&readval));
          }
        }
        readval = readval>>8 & 0xf; /* Extract bits 8-11 */
        break;
      case BX_STAT_CMP_ADDR_HI:
        if (BestXHasMephisto(handle))
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_BUSADDR_HIGH_M,sizeof(bx_int32),&readval));
        }
        else
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_STATUS_M,sizeof(bx_int16),&readval));
          if (readval & 1<<0)
          {
            /* Compare error from datagenerator (only)*/
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_BUSADDR_HIGH_M,sizeof(bx_int32),&readval));
          }
          else
          {
            /* Compare error from datamemory (or no error) */
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_BUSADDR_HIGH_F,sizeof(bx_int32),&readval));
          }
        }
        break;
      case BX_STAT_CMP_ADDR_LO:
        if (BestXHasMephisto(handle))
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_BUSADDR_LOW_M,sizeof(bx_int32),&readval));
        }
        else
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_STATUS_M,sizeof(bx_int16),&readval));
          if (readval & 1<<0)
          {
            /* Compare error from datagenerator (only)*/
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_BUSADDR_LOW_M,sizeof(bx_int32),&readval));
          }
          else
          {
            /* Compare error from datamemory (or no error) */
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_BUSADDR_LOW_F,sizeof(bx_int32),&readval));
          }
        }
        break;
      case BX_STAT_CMP_EXTCMD:
        if (BestXHasMephisto(handle))
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_CMDBE_M,sizeof(bx_int32),&readval));
        }
        else
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_STATUS_M,sizeof(bx_int16),&readval));
          if (readval & 1<<0)
          {
            /* Compare error from datagenerator (only)*/
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_CMDBE_M,sizeof(bx_int32),&readval));
          }
          else
          {
            /* Compare error from datamemory (or no error) */
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_CMDBE_F,sizeof(bx_int32),&readval));
          }
        }
        readval = readval>>12 & 0xf; /* Extract bits 12-15 */
        break;
      case BX_STAT_CMP_ATTR_LO:
        if (BestXHasMephisto(handle))
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_BUSATTR_M,sizeof(bx_int32),&readval));
        }
        else
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_STATUS_M,sizeof(bx_int16),&readval));
          if (readval & 1<<0)
          {
            /* Compare error from datagenerator (only)*/
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_BUSATTR_M,sizeof(bx_int32),&readval));
          }
          else
          {
            /* Compare error from datamemory (or no error) */
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_BUSATTR_F,sizeof(bx_int32),&readval));
          }
        }
        break;
      case BX_STAT_CMP_ACTUAL_HI:
        if (BestXHasMephisto(handle))
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_BUSDATA_HIGH_M,sizeof(bx_int32),&readval));
        }
        else
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_STATUS_M,sizeof(bx_int16),&readval));
          if (readval & 1<<0)
          {
            /* Compare error from datagenerator (only)*/
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_BUSDATA_HIGH_M,sizeof(bx_int32),&readval));
          }
          else
          {
            /* Compare error from datamemory (or no error) */
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_BUSDATA_HIGH_F,sizeof(bx_int32),&readval));
          }
        }
        break;
      case BX_STAT_CMP_ACTUAL_LO:
        if (BestXHasMephisto(handle))
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_BUSDATA_LOW_M,sizeof(bx_int32),&readval));
        }
        else
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_STATUS_M,sizeof(bx_int16),&readval));
          if (readval & 1<<0)
          {
            /* Compare error from datagenerator (only)*/
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_BUSDATA_LOW_M,sizeof(bx_int32),&readval));
          }
          else
          {
            /* Compare error from datamemory (or no error) */
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_BUSDATA_LOW_F,sizeof(bx_int32),&readval));
          }
        }
        break;
      case BX_STAT_CMP_REF_HI:
        if (BestXHasMephisto(handle))
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_CMPDATA_HIGH_M,sizeof(bx_int32),&readval));
        }
        else
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_STATUS_M,sizeof(bx_int16),&readval));
          if (readval & 1<<0)
          {
            /* Compare error from datagenerator (only)*/
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_CMPDATA_HIGH_M,sizeof(bx_int32),&readval));
          }
          else
          {
            /* Compare error from datamemory (or no error) */
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_CMPDATA_HIGH_F,sizeof(bx_int32),&readval));
          }
        }
        break;
      case BX_STAT_CMP_REF_LO:
        if (BestXHasMephisto(handle))
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_CMPDATA_LOW_M,sizeof(bx_int32),&readval));
        }
        else
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_STATUS_M,sizeof(bx_int16),&readval));
          if (readval & 1<<0)
          {
            /* Compare error from datagenerator (only)*/
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_CMPDATA_LOW_M,sizeof(bx_int32),&readval));
          }
          else
          {
            /* Compare error from datamemory (or no error) */
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_CMPDATA_LOW_F,sizeof(bx_int32),&readval));
          }
        }
        break;
      case BX_STAT_CMP_DATAPHASE:
        if (BestXHasMephisto(handle))
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_DPH_M,sizeof(bx_int32),&readval));
        }
        else
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_STATUS_M,sizeof(bx_int16),&readval));
          if (readval & 1<<0)
          {
            /* Compare error from datagenerator (only)*/
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_DPH_M,sizeof(bx_int32),&readval));
          }
          else
          {
            /* Compare error from datamemory (or no error) */
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_DPH_F,sizeof(bx_int32),&readval));
          }
        }
        readval &= 0x7ff; /* Extract bits 0-10 */
        break;
      case BX_STAT_CMP_DATASUBPHASE:
        if (BestXHasMephisto(handle))
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_DPH_M,sizeof(bx_int32),&readval));
        }
        else
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_STATUS_M,sizeof(bx_int16),&readval));
          if (readval & 1<<0)
          {
            /* Compare error from datagenerator (only)*/
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_DPH_M,sizeof(bx_int32),&readval));
          }
          else
          {
            /* Compare error from datamemory (or no error) */
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_DPH_F,sizeof(bx_int32),&readval));
          }
        }
        readval = readval>>11 & 0x3; /* Extract bits 11-12 */
        break;
      case BX_STAT_CMP_RESOURCE:
        if (BestXHasMephisto(handle))
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_STATUS_M,sizeof(bx_int32),&readval));
          readval=readval>>4 & 0x1; /* extract bit 4 (0=datamem, 1=datagen) */
        }
        else
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_STATUS_M,sizeof(bx_int16),&readval));
          if (readval & 1<<0)
          {
            /* Compare error from datagenerator (only)*/
            readval=1;
          }
          else
          {
            /* Compare error from datamemory (or no error) */
            readval=0;
          }
        }
        break;
      case BX_STAT_CMP_BEATTR:
        if (BestXHasMephisto(handle))
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_CMDBE_M,sizeof(bx_int32),&readval));
        }
        else
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_STATUS_M,sizeof(bx_int16),&readval));
          if (readval & 1<<0)
          {
            /* Compare error from datagenerator (only)*/
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_CMDBE_M,sizeof(bx_int32),&readval));
          }
          else
          {
            /* Compare error from datamemory (or no error) */
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_CMDBE_F,sizeof(bx_int32),&readval));
          }
        }
        readval = readval>>16 & 0xf; /* Extract bits 16-19 */
        break;
      case BX_STAT_CMP_BEDATA:
        if (BestXHasMephisto(handle))
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_CMDBE_M,sizeof(bx_int32),&readval));
        }
        else
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_STATUS_M,sizeof(bx_int16),&readval));
          if (readval & 1<<0)
          {
            /* Compare error from datagenerator (only)*/
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_CMDBE_M,sizeof(bx_int32),&readval));
          }
          else
          {
            /* Compare error from datamemory (or no error) */
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_CMDBE_F,sizeof(bx_int32),&readval));
          }
        }
        readval &= 0xff; /* Extract bits 0-7 */
        break;
      case BX_STAT_CMP_XFERSIZE:
        if (BestXHasMephisto(handle))
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_STATUS_M,sizeof(bx_int32),&readval));
        }
        else
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_STATUS_M,sizeof(bx_int16),&readval));
          if (readval & 1<<0)
          {
            /* Compare error from datagenerator (only)*/
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_STATUS_M,sizeof(bx_int32),&readval));
          }
          else
          {
            /* Compare error from datamemory (or no error) */
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_STATUS_F,sizeof(bx_int32),&readval));
          }
        }
        readval = readval>>3 & 0x1; /* Extract bit 3 */
        break;
      case BX_STAT_CMP_ICMP:
        if (BestXHasMephisto(handle))
        {
          /* not working correct for mephisto !!! */
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_STATUS_M,sizeof(bx_int32),&readval));
        }
        else
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_STATUS_M,sizeof(bx_int16),&readval));
          if (readval & 1<<0)
          {
            /* Compare error from datagenerator (only)*/
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_STATUS_M,sizeof(bx_int32),&readval));
          }
          else
          {
            /* Compare error from datamemory (or no error) */
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_STATUS_F,sizeof(bx_int32),&readval));
          }
        }
        readval = readval>>1 & 0x1; /* Extract bit 1 */
        break;
      case BX_STAT_CMP_TCMP:
        if (BestXHasMephisto(handle))
        {
          /* not working correct for mephisto !!! */
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_STATUS_M,sizeof(bx_int32),&readval));
        }
        else
        {
          BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_STATUS_M,sizeof(bx_int16),&readval));
          if (readval & 1<<0)
          {
            /* Compare error from datagenerator (only)*/
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_STATUS_M,sizeof(bx_int32),&readval));
          }
          else
          {
            /* Compare error from datamemory (or no error) */
            BX_TRY(BestXDirectRegRead(handle,BX_REG_ERR_STATUS_F,sizeof(bx_int32),&readval));
          }
        }
        readval = readval>>2 & 0x1; /* Extract bit 2 */
        break;
      

      /***********************************************************************
       *********************  DIAGNOSTICS  ***********************************
       **********************************************************************/

      /* Caution:
         If you poll one of these properties repeatedly, make sure to leave
         at least 0.6 seconds time between subsequent calls.
         Else the read values wont be updated properly.
       */

      case BX_STAT_DIAG_FPGATEMP:

        /* This property only exists for E2930A (FPGA) (handled by dyncaps)! */

        /* Read "Faust's" temperature from the 1618 (8 bit two's complement) */

        /* Put address of 1618 into address register */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_ADDRESS,sizeof(bx_int16),0x18));

        /* Remote temp. register address in command reg. (01h) */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_COMMAND,sizeof(bx_int16),0x01));

        /* Start read transfer in control reg. */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_CONTROL,sizeof(bx_int16),0x03));

        /* Wait for xfer to finish */
        BX_TRY(BestXWaitForDiagXfer(handle));

        /* Read data register, it now contains the temperature (8 bit two's complement)  */
        BX_TRY(BestXDirectRegRead(handle,BX_REG_DIAG_DATA,sizeof(bx_int16),&readval));
        readval&=0xff;
        break;
      case BX_STAT_DIAG_BOARDTEMP:
        /* Read LM87 temperature (8 bit two's complement) */

        /* Put address of LM87 into address register */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_ADDRESS,sizeof(bx_int16),0x2D));

        /* Read internal temperature (27h) (8 bit two's complement) */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_COMMAND,sizeof(bx_int16),0x27));

        /* Read from slave and check transfer result */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_CONTROL,sizeof(bx_int16),0x03));
        /* Wait for xfer to finish */
        BX_TRY(BestXWaitForDiagXfer(handle));

        /* Read received data */
        BX_TRY(BestXDirectRegRead(handle,BX_REG_DIAG_DATA,sizeof(bx_int16),&readval));
        readval&=0xff;
        break;
      case BX_STAT_DIAG_V5:
        /* Put address of LM87 into address register */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_ADDRESS,sizeof(bx_int16),0x2D));

        /* Read +5V (23h) (LSB size 26 mV) */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_COMMAND,sizeof(bx_int16),0x23));

        /* Read from slave and check transfer result */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_CONTROL,sizeof(bx_int16),0x03));
        /* Wait for xfer to finish */
        BX_TRY(BestXWaitForDiagXfer(handle));

        /* Read received data */
        BX_TRY(BestXDirectRegRead(handle,BX_REG_DIAG_DATA,sizeof(bx_int16),&readval));
        readval&=0xff;
        readval*=26; /* Convert into mV */
        break;
      case BX_STAT_DIAG_V12:

        /* Put address of LM87 into address register */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_ADDRESS,sizeof(bx_int16),0x2D));

        /* Read +12V (24h) (LSB size 62.5 mV) */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_COMMAND,sizeof(bx_int16),0x24));

        /* Read from slave and check transfer result */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_CONTROL,sizeof(bx_int16),0x03));
        /* Wait for xfer to finish */
        BX_TRY(BestXWaitForDiagXfer(handle));

        /* Read received data */
        BX_TRY(BestXDirectRegRead(handle,BX_REG_DIAG_DATA,sizeof(bx_int16),&readval));
        readval&=0xff;
        readval=(bx_int32)(readval*62.5+0.5); /* Convert into mV */
        break;
      case BX_STAT_DIAG_VCORE:

        /* Put address of LM87 into address register */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_ADDRESS,sizeof(bx_int16),0x2D));

        /* Read Ain2 (29h) (LSB size 9.8 mV) */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_COMMAND,sizeof(bx_int16),0x29));

        /* Read from slave and check transfer result */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_CONTROL,sizeof(bx_int16),0x03));
        /* Wait for xfer to finish */
        BX_TRY(BestXWaitForDiagXfer(handle));

        /* Read received data */
        BX_TRY(BestXDirectRegRead(handle,BX_REG_DIAG_DATA,sizeof(bx_int16),&readval));
        readval&=0xff;
        readval=(bx_int32)(readval*9.8+0.5); /* Convert into mV */
        break;
      case BX_STAT_DIAG_VCC:

        /* Put address of LM87 into address register */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_ADDRESS,sizeof(bx_int16),0x2D));

        /* Read +Vcc (22h) (LSB size 17.2 mV) */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_COMMAND,sizeof(bx_int16),0x22));

        /* Read from slave and check transfer result */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_CONTROL,sizeof(bx_int16),0x03));
        /* Wait for xfer to finish */
        BX_TRY(BestXWaitForDiagXfer(handle));

        /* Read received data */
        BX_TRY(BestXDirectRegRead(handle,BX_REG_DIAG_DATA,sizeof(bx_int16),&readval));
        readval&=0xff;
        readval=(bx_int32)(readval*17.2+0.5); /* Convert into mV */
        break;
      case BX_STAT_DIAG_VIO:

        /* Put address of LM87 into address register */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_ADDRESS,sizeof(bx_int16),0x2D));

        /* Read Vccp1 (21h) (LSB size 14.1 mV) */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_COMMAND,sizeof(bx_int16),0x21));

        /* Read from slave and check transfer result */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_CONTROL,sizeof(bx_int16),0x03));
        /* Wait for xfer to finish */
        BX_TRY(BestXWaitForDiagXfer(handle));

        /* Read received data */
        BX_TRY(BestXDirectRegRead(handle,BX_REG_DIAG_DATA,sizeof(bx_int16),&readval));
        readval&=0xff;
        readval=(bx_int32)(readval*14.1+0.5); /* Convert into mV */
        break;
      case BX_STAT_DIAG_VCCP2:

        /* Put address of LM87 into address register */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_ADDRESS,sizeof(bx_int16),0x2D));

        /* Read +Vccp2 (25h) (LSB size 14.1 mV) */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_COMMAND,sizeof(bx_int16),0x25));

        /* Read from slave and check transfer result */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_CONTROL,sizeof(bx_int16),0x03));
        /* Wait for xfer to finish */
        BX_TRY(BestXWaitForDiagXfer(handle));

        /* Read received data */
        BX_TRY(BestXDirectRegRead(handle,BX_REG_DIAG_DATA,sizeof(bx_int16),&readval));
        readval&=0xff;

        readval=(bx_int32)(readval*14.1+0.5); /* Convert into mV */


        /* PCI -12 Volt Rail = VCCP2 - (196000 * ( (V5 - VCCP2) / 38300 - VCCP2 / 130000) ) */
        BX_TRY(BestXStatusRead(handle,BX_STAT_DIAG_V5,&t));
        readval=(bx_int32) -(readval-(196000.0*((t-readval)/38300.0-readval/130000.0)));
        /* Caution: We return the negated (i.e. positive) value here ! */
        break;

      case BX_STAT_DIAG_VBOARD:

        /* Put address of LM87 into address register */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_ADDRESS,sizeof(bx_int16),0x2D));

        /* Read +2.5 V from +2.5Vin (20h) ( LSB size 13 mV) */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_COMMAND,sizeof(bx_int16),0x20));

        /* Read from slave and check transfer result */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_CONTROL,sizeof(bx_int16),0x03));
        /* Wait for xfer to finish */
        BX_TRY(BestXWaitForDiagXfer(handle));

        /* Read received data */
        BX_TRY(BestXDirectRegRead(handle,BX_REG_DIAG_DATA,sizeof(bx_int16),&readval));
        readval&=0xff;
        readval*=13; /* Convert into mV */
        break;
      case BX_STAT_DIAG_STATUS:
        /* Read the status-byte: shows busy converting,
           high or low temp. alert active and most important diode fault (Datasheet) */

        /* Put address of 1618 into address register */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_ADDRESS,sizeof(bx_int16),0x18));

        /* Status-byte register address in command reg. */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_COMMAND,sizeof(bx_int16),0x02));

        /* Start read transfer in control reg. */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_CONTROL,sizeof(bx_int16),0x03));

        /* Wait for xfer to finish */
        BX_TRY(BestXWaitForDiagXfer(handle));

        /* Read data register, it now contains the temperature (8 bit two's complement)  */
        BX_TRY(BestXDirectRegRead(handle,BX_REG_DIAG_DATA,sizeof(bx_int16),&readval));
        readval&=0xff;
        break;
      case BX_STAT_DIAG_FANSPEED: /* Fanspeed in RPS (should be in [90-100]) */

        /* Read Fan1 count (28h) (value count is internally calculated:
           count= 1,350,000/(RPM*Divisor from Fan Divisor Reg.)) (Fan divisor above set to 1)
           Apparently a 3 tacho pulses per revolution fan: 
           RPM = 1,350,000/(count * 1 * 3)
           RPS = 1,350,000/(count * 1 * 3 * 60) = 7500/count
         */

        /* Put address of LM87 into address register */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_ADDRESS,sizeof(bx_int16),0x2D));

        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_COMMAND,sizeof(bx_int16),0x28));

        /* Read from slave and check transfer result */
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_DIAG_CONTROL,sizeof(bx_int16),0x03));
        /* Wait for xfer to finish */
        BX_TRY(BestXWaitForDiagXfer(handle));

        /* Read received data */
        BX_TRY(BestXDirectRegRead(handle,BX_REG_DIAG_DATA,sizeof(bx_int16),&readval));
        readval&=0xff;

        if (readval==0xff)
        {
          /* Counter overflow.
             With a divisor of 1 this happens when RPS = [0,..,30]
           */
          readval=0;
        }
        else
        {
          readval=(bx_int32)floor(7500.0/readval+0.5);
        }
        break;

      default:
        BX_TRY_ERROR(BX_E_INVALID_CASE);
        break;
    }
    
    *val = readval;
  }

  BX_ERRETURN(BX_TRY_RET);
}

/********************************************************************
  Best Status *******************************************************
 *******************************************************************/

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXStatusDump(
 *
 * Purpose: Write a status report 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXStatusDump( 
  bx_handletype  handle,
  bx_charptrtype filename

)
{
  BX_DECLARE_FUNCNAME("BestXStatusDump [statusdump]");
  BX_TRY_VARS_NO_PROG;

  FILE *fp;
  char *info;
  bx_int32 val,i,j;
  
  fp=BESTX_FOPEN(filename,"w");

  if (fp==NULL)
  {
    return BX_E_FILE_OPEN;
  }

  BX_TRY_BEGIN
  {
    BESTX_FPRINTF(fp,"********************\n*** BestX Status ***\n********************\n\n");

    /* run mode */
    BX_TRY(BestXDirectRegRead(handle,BX_REG_EX_CLK_SWITCH_STAT_REG,sizeof(bx_int16),&val));
    BESTX_FPRINTF(fp,"Exerciser is in %s-mode\n",val&1?"RUN":"PROGRAMMING");
    BX_TRY(BestXDirectRegRead(handle,BX_REG_TR_CLK_SWITCH_STAT_REG,sizeof(bx_int16),&val));
    BESTX_FPRINTF(fp,"Analyzer is in %s-mode\n",val&1?"RUN":"PROGRAMMING");
    BX_TRY(BestXDirectRegRead(handle,BX_REG_AN_CLK_SWITCH_STAT_REG,sizeof(bx_int16),&val));
    BESTX_FPRINTF(fp,"Performance is in %s-mode\n\n",val&1?"RUN":"PROGRAMMING");

    BESTX_FPRINTF(fp,"handle->CardIsInProgrammingMode=%lu\n\n",bx_handlearray[handle].CardIsInProgrammingMode);

    /* Reset code */
    BX_TRY(BestXStatusRead(handle,BX_STAT_RESETCODE,&val));
    BESTX_FPRINTF(fp,"Resetcode: 0x%01lx\n\n",val);

    /* PCI mode */
    BX_TRY(BestXDirectRegRead(handle,BX_REG_PCIX_MEN_REG,sizeof(bx_int16),&val));

    switch (val)
    {
      case 0x2:
      case 0xa:
        BESTX_FPRINTF(fp,"Card is in PCI mode (forced): 0x%04lx\n",val);
        break;
      case 0x7:
      case 0xf:
        BESTX_FPRINTF(fp,"Card is in PCI-X mode (forced): 0x%04lx\n",val);
        break;
      case 0x0:
      case 0x1:
        BESTX_FPRINTF(fp,"Card is in PCI mode (from #RST): 0x%04lx\n",val);
        break;
      case 0xd:
      case 0xc:
        BESTX_FPRINTF(fp,"Card is in PCI-X mode (from #RST): 0x%04lx\n",val);
        break;
      default:
        BESTX_FPRINTF(fp,"Card's PCI(X) mode unknown: 0x%04lx\n",val);
        break;
    }

    /* Retry bits */
    BX_TRY(BestXDirectRegRead(handle,BX_REG_CNF_DO_RETRY_BIT_REG,sizeof(bx_int16),&val));
    BESTX_FPRINTF(fp,"Config Retry bit %s\n",val?"set":"not set");
    BX_TRY(BestXDirectRegRead(handle,BX_REG_FORCE_RETRY_SYNC_BIT_REG,sizeof(bx_int16),&val));
    BESTX_FPRINTF(fp,"Target Retry bit %s\n",val?"set":"not set");

    /* Decoder enable register */
    BX_TRY(BestXDirectRegRead(handle,BX_REG_DEC_EN_SREG,sizeof(bx_int16),&val));
    BESTX_FPRINTF(fp,"\nDecoder enable register: 0x%08lx\n",val);
    BESTX_FPRINTF(fp,"STD0: %s\n",(val&1<<0?"Enabled":"Disabled"));
    BESTX_FPRINTF(fp,"STD1: %s\n",(val&1<<1?"Enabled":"Disabled"));
    BESTX_FPRINTF(fp,"STD2: %s\n",(val&1<<2?"Enabled":"Disabled"));
    BESTX_FPRINTF(fp,"STD3: %s\n",(val&1<<3?"Enabled":"Disabled"));
    BESTX_FPRINTF(fp,"STD4: %s\n",(val&1<<4?"Enabled":"Disabled"));
    BESTX_FPRINTF(fp,"STD5: %s\n",(val&1<<5?"Enabled":"Disabled"));
    BESTX_FPRINTF(fp,"EXPR: %s\n",(val&1<<6?"Enabled":"Disabled"));
    BESTX_FPRINTF(fp,"REQT: %s\n",(val&1<<7?"Enabled":"Disabled"));
    BESTX_FPRINTF(fp,"CNFG: %s\n",(val&1<<8?"Enabled":"Disabled"));
    
    /* Split decoder */
    BESTX_FPRINTF(fp,"\nSplit decoders:\n");
    for (i=0;i<BX_CTSPLITPROP_DEPTH;i++)      /* For each of our four split decoders */
    {
      /* Mask */
      BX_TRY(BestXCTSplitCondGet(handle,i,BX_CTSPLIT_ADDRMASK_HI,&val)); 
      BESTX_FPRINTF(fp,"ADDRMASK[%lu] = 0x%08lx",i,val);
      BX_TRY(BestXCTSplitCondGet(handle,i,BX_CTSPLIT_ADDRMASK_LO,&val)); 
      BESTX_FPRINTF(fp," %08lx\n",val);

      /* Value */
      BX_TRY(BestXCTSplitCondGet(handle,i,BX_CTSPLIT_ADDRVAL_HI,&val)); 
      BESTX_FPRINTF(fp,"ADDRVAL[%lu]  = 0x%08lx",i,val);
      BX_TRY(BestXCTSplitCondGet(handle,i,BX_CTSPLIT_ADDRVAL_LO,&val)); 
      BESTX_FPRINTF(fp," %08lx\n",val);

      /* Commands */
      BX_TRY(BestXCTSplitCondGet(handle,i,BX_CTSPLIT_CMDS,&val)); 
      BESTX_FPRINTF(fp,"CMDS[%lu] = 0x%08lx\n",i,val);

      /* Decoders */
      BX_TRY(BestXCTSplitCondGet(handle,i,BX_CTSPLIT_DEC,&val)); 
      BESTX_FPRINTF(fp,"DEC[%lu] = 0x%08lx\n",i,val);

      /* Queues */
      BX_TRY(BestXCTSplitCondGet(handle,i,BX_CTSPLIT_QUEUE,&val)); 
      BESTX_FPRINTF(fp,"QUEUE[%lu] = 0x%08lx\n",i,val);
    }

    /* Status/Command register */
    BX_TRY(BestXDirectRegRead(handle,0x405404,sizeof(bx_int32),&val));
    BESTX_FPRINTF(fp,"\nStatus/Command register: 0x%08lx\n",val);
    BESTX_FPRINTF(fp,"Master (RI) %s enabled\n",(val&4) ?"":"not");
    BESTX_FPRINTF(fp,"IO space %s enabled\n",(val&1) ?"":"not");
    BESTX_FPRINTF(fp,"Mem space %s enabled\n",(val&2) ?"":"not");
        

    /* Pattern terms */
    BESTX_FPRINTF(fp,"\nPattern terms:\n");
    BX_TRY(BestXPattGet(handle,BX_PATT_BUS0,&info));
    BESTX_FPRINTF(fp,"BUS0: \"%s\"\n",info);
    BX_TRY(BestXPattGet(handle,BX_PATT_BUS1,&info));
    BESTX_FPRINTF(fp,"BUS1: \"%s\"\n",info);
    if (BestXHasFaust(handle))
    {
      BX_TRY(BestXPattGet(handle,BX_PATT_BUS2,&info));
      BESTX_FPRINTF(fp,"BUS2: \"%s\"\n",info);
      BX_TRY(BestXPattGet(handle,BX_PATT_BUS3,&info));
      BESTX_FPRINTF(fp,"BUS3: \"%s\"\n",info);
    }
    BX_TRY(BestXPattGet(handle,BX_PATT_OBS0,&info));
    BESTX_FPRINTF(fp,"OBS0: \"%s\"\n",info);
    BX_TRY(BestXPattGet(handle,BX_PATT_OBS1,&info));
    BESTX_FPRINTF(fp,"OBS1: \"%s\"\n",info);
    BX_TRY(BestXPattGet(handle,BX_PATT_OBS2,&info));
    BESTX_FPRINTF(fp,"OBS2: \"%s\"\n",info);
    BX_TRY(BestXPattGet(handle,BX_PATT_OBS3,&info));
    BESTX_FPRINTF(fp,"OBS3: \"%s\"\n",info);
    BX_TRY(BestXPattGet(handle,BX_PATT_ERR0,&info));
    if (BestXHasFaust(handle))
    {
      BX_TRY(BestXPattGet(handle,BX_PATT_OBS4,&info));
      BESTX_FPRINTF(fp,"OBS4: \"%s\"\n",info);
      BX_TRY(BestXPattGet(handle,BX_PATT_OBS5,&info));
      BESTX_FPRINTF(fp,"OBS5: \"%s\"\n",info);
    }
    BESTX_FPRINTF(fp,"ERR0: \"%s\"\n",info);
    BX_TRY(BestXPattGet(handle,BX_PATT_COND0,&info));
    BESTX_FPRINTF(fp,"COND0: \"%s\"\n",info);
    BX_TRY(BestXPattGet(handle,BX_PATT_COND1,&info));
    BESTX_FPRINTF(fp,"COND1: \"%s\"\n",info);
    BX_TRY(BestXPattGet(handle,BX_PATT_TRAN0,&info));
    BESTX_FPRINTF(fp,"TRAN0: \"%s\"\n",info);

    /* Trigger sequencer */
    BESTX_FPRINTF(fp,"\nTrigger sequencer:\n");
    BX_TRY(BestXTrigCondGet(handle,0,BX_TRIGCOND_TRIG,&info));
    BESTX_FPRINTF(fp,"TRIG: \"%s\"\n",info);
    BX_TRY(BestXTrigCondGet(handle,0,BX_TRIGCOND_SQ,&info));
    BESTX_FPRINTF(fp,"SQ: \"%s\"\n",info);

    /* Powerup loader code */
    BX_TRY(BestXLoaderDecode(handle,"c:\\bx_pucode.txt",BX_LOADER_IMAGE));
    BESTX_FPRINTF(fp,"\nPowerup loader:\n");
    BX_TRY(BestXDirectRegRead(handle,BX_LOADER_IMAGE,sizeof(bx_int16),&val));
    BESTX_FPRINTF(fp,"Powerup loader code (see c:\\bx_pucode.txt) %spresent\n",val==0xffff?"not ":"");
    BX_TRY(BestXDirectRegRead(handle,BX_REG_RES_LOCK_REG,sizeof(bx_int32),&val));
    BESTX_FPRINTF(fp,"Powerup loader has %sexecuted (0x%08lx)\n",(val&BX_PU_COUNTERMASK?"":"not "),val);
    BX_TRY(BestXDirectRegRead(handle,BX_REG_DEFLOAD_PROG_START_REG,sizeof(bx_int32),&val));
    BESTX_FPRINTF(fp,"Powerup loader start pointer: 0x%08lx (should be 0x003ffffc)\n",val);
    BX_TRY(BestXDirectRegRead(handle,0x3ffffc,sizeof(bx_int32),&val));
    BESTX_FPRINTF(fp,"Powerup loader start address: 0x%08lx (should be 0x%08lx)\n",val,BX_LOADER_IMAGE);
    BX_TRY(BestXDirectRegRead(handle,BX_REG_DEFLOAD_CURR_PROG_CNTR_REG,sizeof(bx_int32),&val));
    BESTX_FPRINTF(fp,"Powerup loader program counter is: 0x%08lx (should be > 0x100000)\n",val);
    BX_TRY(BestXDirectRegRead(handle,BX_REG_DBI_STATUS_REG,sizeof(bx_int16),&val));
    BESTX_FPRINTF(fp,"Powerup loader status is: 0x%04x (should be 0xc000)\n",val);
    BX_TRY(BestXDirectRegRead(handle,BX_REG_DBI_CMD_REG,sizeof(bx_int16),&val));
    BESTX_FPRINTF(fp,"Powerup loader will %srun with PCIRST# assertion\n",(val&1?"":"not (!!)"));

    BX_TRY(BestXStatusRead(handle,BX_STAT_BUSSPEED,&val));
    BESTX_FPRINTF(fp,"\nBusSpeed: %lu Hz",val);
    BX_TRY(BestXStatusRead(handle,BX_STAT_PLLSPEED,&val));
    BESTX_FPRINTF(fp,"\nPLLSpeed: %lu Hz",val);
    BX_TRY(BestXDirectRegRead(handle,BX_REG_SYS_CLK_MODE_REG,sizeof(bx_int16),&val));
    BESTX_FPRINTF(fp,"\nPLLStatus: 0x%04lx\n",val);
    
    /* Relays (30/23er only) */
    if (!BestXHasMephisto(handle))
    {
      /* Check, wether Flash contains a valid relay value */
      BX_TRY(BestXDirectRegRead(handle,BX_RELAY_STATUS,sizeof(bx_int16),&val));
      BESTX_FPRINTF(fp,"\nRelay status: 0x%04lx ",val);
      if ((val&0xf0)==(0xf0& ~(val<<4))) /* Check complement (0xffff e.al. will surely fail here !) */
      {
        switch (val & 0x70) /* Check bits 4,5,6 (i.e. relays A,B,C) only */
        {
          case 1<<4:
            /* Relay A set */  
            BESTX_FPRINTF(fp,"(PCIX266)\n");         
            break;
          case 1<<5:
            /* Relay B set */  
            BESTX_FPRINTF(fp,"(PCI)\n");         
            break;
          case 1<<6:
            /* Relay C set */  
            BESTX_FPRINTF(fp,"(PCIX66)\n");         
            break;
          case 0:
            /* Relays A,B,C unset */
            BESTX_FPRINTF(fp,"(PCIX133)\n");         
            break;
          case 1<<4|1<<5:
            /* Relays A,B set */
            BESTX_FPRINTF(fp,"(PCIX533)\n");         
            break;
          default:
            BESTX_FPRINTF(fp,"(ERROR: Please execute BestXBoardProg() once and retry)\n");
            break;
        }
      }
      else
      {
        BESTX_FPRINTF(fp,"(ERROR: Please execute BestXBoardProg() once and retry)\n");
      }
    }

    /* Jumper settings */
    BESTX_FPRINTF(fp,"\nBig DIP switch S101:\n");

    BX_TRY(BestXDirectRegRead(handle,BX_REG_GEN_DEC_PROP_REG,sizeof(bx_int16),&val));
    BESTX_FPRINTF(fp,"S1(VISIBLE):     %s\n",(val&2?"OFF":"ON"));
    BX_TRY(BestXDirectRegRead(handle,BX_REG_DBI_STATUS_REG,sizeof(bx_int16),&val));
    BESTX_FPRINTF(fp,"S2(USR SETTING): %s\n",(val&1<<15?"ON":"OFF"));

    if (BestXHasFirmware(handle)) /* Need Xilinx here ! */
    {
      BX_TRY(BestXDirectRegRead(handle,0xfff906,sizeof(bx_int16),&val));
      BESTX_FPRINTF(fp,"S3(STD/RECOVR):  %s\n", (val&0x80 ? "ON":"OFF")); /* pdr[7] */
      BESTX_FPRINTF(fp,"S4(AUX0):        %s\n","TBD: Xilinx access");
      BESTX_FPRINTF(fp,"S5(AUX1):        %s\n", (val&0x40 ? "ON":"OFF")); /* pdr[6] */
      BESTX_FPRINTF(fp,"S6(AUX2):        %s\n", (val&0x20 ? "ON":"OFF")); /* pdr[5] */
      BESTX_FPRINTF(fp,"S7(AUX3):        %s\n", (val&0x10 ? "ON":"OFF")); /* pdr[4] */
      BESTX_FPRINTF(fp,"S8(AUX4):        %s\n", (val&0x01 ? "ON":"OFF")); /* pdr[0] */
    }

    if (BestXHasMephisto(handle))
    {
      BX_TRY(BestXDirectRegRead(handle,BX_REG_SYS_CLK_MODE_REG,sizeof(bx_int16),&val));
      /* Extract Bits 12-15 */
      val=val>>12 & 0xf;

      BESTX_FPRINTF(fp,"\n\nSmall DIP switch S102: 0x%lx i.e. ",val);
      
      if (val&0x1 && val&0x2)
      {
        BESTX_FPRINTF(fp,"Auto range select\n");
      }
      else if (val&0x1 && val&0x4 && val&0x8)
      {
        BESTX_FPRINTF(fp,"Reserved\n");
      }
      else if (val&0x1 && val&0x4)
      {
        BESTX_FPRINTF(fp,"PLL range 75-133 MHz\n");
      }
      else if (val&0x1 && val&0x8)
      {
        BESTX_FPRINTF(fp,"PLL range 50-88 MHz\n");
      }
      else if (val&0x1)
      {
        BESTX_FPRINTF(fp,"PLL range 37-66 MHz\n");
      }
      else if (!(val&0x1))
      {
        BESTX_FPRINTF(fp,"PLL off\n");
      }
      else
      {
        /* should never happen, error in previous lines (?) */
        BESTX_FPRINTF(fp,"State unknown\n");
      }            
      BESTX_FPRINTF(fp,"S1:              %s\n",(val&1 ? "ON":"OFF"));
      BESTX_FPRINTF(fp,"S2:              %s\n",(val&2 ? "ON":"OFF"));
      BESTX_FPRINTF(fp,"S3:              %s\n",(val&4 ? "ON":"OFF"));
      BESTX_FPRINTF(fp,"S4:              %s\n",(val&8 ? "ON":"OFF"));
    }

    /* Version Info */
    BESTX_FPRINTF(fp,"\n\nVERSION INFO\n",info);

    BX_TRY(BestXVersionRead(handle,BX_VERSION_CAPI,&info));
    BESTX_FPRINTF(fp,"CAPI version: %s\n",info);
    BX_TRY(BestXVersionRead(handle,BX_VERSION_FIRMWARE,&info));
    BESTX_FPRINTF(fp,"BIOS version: %s\n",info);
    BX_TRY(BestXVersionRead(handle,BX_VERSION_CORE,&info));
    BESTX_FPRINTF(fp,"Core version: %s\n",info);
    BX_TRY(BestXVersionRead(handle,BX_VERSION_XILINX,&info));
    BESTX_FPRINTF(fp,"Xilinx version: %s\n",info);
    BX_TRY(BestXVersionRead(handle,BX_VERSION_PRODUCT,&info));
    BESTX_FPRINTF(fp,"Product: %s\n",info);
    BX_TRY(BestXVersionRead(handle,BX_VERSION_SERIALNO,&info));
    BESTX_FPRINTF(fp,"Serial: %s\n",info);
    BX_TRY(BestXVersionRead(handle,BX_VERSION_BOARDID,&info));
    BESTX_FPRINTF(fp,"Board: %s\n",info);
    BX_TRY(BestXVersionRead(handle,BX_VERSION_MEPHISTO,&info));
    BESTX_FPRINTF(fp,"Mephisto: %s\n",info);
    BX_TRY(BestXVersionRead(handle,BX_VERSION_FAUST,&info));
    BESTX_FPRINTF(fp,"Faust: %s\n",info);
    if (BestXHasFaustFPGA(handle))
    {
      /* Version, Date and Time of the four Faust images */
      for (i=0;i<4;i++)
      {
        BX_TRY(BestXDirectRegFaustFlashRead(handle,i*0x400000+BX_FAUST_IMAGE_SIZE,&val));
        BESTX_FPRINTF(fp,"Image %lu: %02lx.%02lx",i,val>>24&0xff,val>>16&0xff);
        BESTX_FPRINTF(fp," Date: %02lx/%02lx",val>>8&0xff,val&0xff); /* MM/DD */
 
        BX_TRY(BestXDirectRegFaustFlashRead(handle,i*0x400000+BX_FAUST_IMAGE_SIZE+4,&val));
        BESTX_FPRINTF(fp,"/%04lx Time: %02lx:%02lx\n",val>>16&0xffff,val>>8&0xff,val&0xff); /* YYYY HH:MM*/
      }
    }  
    BX_TRY(BestXVersionRead(handle,BX_VERSION_ALTERA,&info));
    BESTX_FPRINTF(fp,"DeepTrace: %s\n",info);
    BX_TRY(BestXVersionRead(handle,BX_VERSION_FIRMWARE_DATE,&info));
    BESTX_FPRINTF(fp,"BiosDate: %s\n",info);

    
    BESTX_FPRINTF(fp,"\n\tConfig Space (on card)\n");

    for (i=0;i<=0x7c-0x20;i+=4) /* 0x20 = Skip the PCI only part !! */
    {
      BESTX_FPRINTF(fp,"\nMask   [0x%2lx]: ",(i<=0x3c?i:i+0x20));
      BX_TRY(BestXDirectRegRead(handle,0x405480+i,sizeof(bx_int32),&val));
      for(j=0;j<32;j++)
      {
        BESTX_FPRINTF(fp,"%c",(1<<(31-j) & val?'1':'0'));
      }
      BESTX_FPRINTF(fp," = 0x%08lx",val);

      BESTX_FPRINTF(fp,"\nValue  [0x%2lx]: ",(i<=0x3c?i:i+0x20));
      BX_TRY(BestXDirectRegRead(handle,0x405400+i,sizeof(bx_int32),&val));
      for(j=0;j<32;j++)
      {
        BESTX_FPRINTF(fp,"%c",(1<<(31-j) & val?'1':'0'));
      }
      BESTX_FPRINTF(fp," = 0x%08lx",val);

    } /* for */
 
    BESTX_FPRINTF(fp,"\n********************\n*** Block memory (on card) ***\n********************\n\n");

    BX_TRY(BestXExerciserProgMode(handle,1));

    /* Set min/max/ctr register */
    BX_TRY(BestXDirectRegWrite(handle,0x404004,sizeof(bx_int16),0));
    BX_TRY(BestXDirectRegWrite(handle,0x404006,sizeof(bx_int16),BX_RIBLK_MEMWIDTH-1));
    BX_TRY(BestXDirectRegWrite(handle,0x404008,sizeof(bx_int32),0));

    for (i=0;i<BX_RIBLK_MEMDEPTH;i++)
    {
      for (j=0;j<BX_RIBLK_MEMWIDTH;j++)
      {
        BX_TRY(BestXDirectRegRead(handle,0x404000,sizeof(bx_int32),&val));
        BESTX_FPRINTF(fp,"%08lx  ",val);
      }
      BESTX_FPRINTF(fp,"\n",val);
    }
    
    BESTX_FPRINTF(fp,"\n********************\n*** RIBeh memory (on card) ***\n********************\n\n");


    /* Set min/max/ctr register */
    BX_TRY(BestXDirectRegWrite(handle,0x404104,sizeof(bx_int16),0));
    BX_TRY(BestXDirectRegWrite(handle,0x404106,sizeof(bx_int16),BX_RIBEH_MEMWIDTH-1));
    BX_TRY(BestXDirectRegWrite(handle,0x404108,sizeof(bx_int32),0));

    for (i=0;i<BX_RIBEH_MEMDEPTH;i++)
    {
      for (j=0;j<BX_RIBEH_MEMWIDTH;j++)
      {
        BX_TRY(BestXDirectRegRead(handle,0x404100,sizeof(bx_int16),&val));
        BESTX_FPRINTF(fp,"%04lx  ",val);
      }
      BESTX_FPRINTF(fp,"\n",val);
    }

    BESTX_FPRINTF(fp,"\n********************\n*** CTBeh memory (on card) ***\n********************\n\n");

    /* Set min/max/ctr register */
    BX_TRY(BestXDirectRegWrite(handle,0x404604,sizeof(bx_int16),0));
    BX_TRY(BestXDirectRegWrite(handle,0x404606,sizeof(bx_int16),BX_CTBEH_MEMWIDTH-1));
    BX_TRY(BestXDirectRegWrite(handle,0x404608,sizeof(bx_int32),0));

    for (i=0;i<BX_CTBEH_MEMDEPTH;i++)
    {
      for (j=0;j<BX_CTBEH_MEMWIDTH;j++)
      {
        BX_TRY(BestXDirectRegRead(handle,0x404600,sizeof(bx_int16),&val));
        BESTX_FPRINTF(fp,"%04lx  ",val);
      }
      BESTX_FPRINTF(fp,"\n",val);
    }

    BESTX_FPRINTF(fp,"\n********************\n*** CIBeh memory (on card) ***\n********************\n\n");

    /* Set min/max/ctr register */
    BX_TRY(BestXDirectRegWrite(handle,0x404204,sizeof(bx_int16),0));
    BX_TRY(BestXDirectRegWrite(handle,0x404206,sizeof(bx_int16),BX_CIBEH_MEMWIDTH-1));
    BX_TRY(BestXDirectRegWrite(handle,0x404208,sizeof(bx_int32),0));

    for (i=0;i<BX_CIBEH_MEMDEPTH;i++)
    {
      for (j=0;j<BX_CIBEH_MEMWIDTH;j++)
      {
        BX_TRY(BestXDirectRegRead(handle,0x404200,sizeof(bx_int16),&val));
        BESTX_FPRINTF(fp,"%04lx  ",val);
      }
      BESTX_FPRINTF(fp,"\n",val);
    }

    BESTX_FPRINTF(fp,"\n********************\n*** RTBeh memory (on card) ***\n********************\n\n");

    /* Set min/max/ctr register */
    BX_TRY(BestXDirectRegWrite(handle,0x404504,sizeof(bx_int16),0));
    BX_TRY(BestXDirectRegWrite(handle,0x404506,sizeof(bx_int16),BX_RTBEH_MEMWIDTH-1));
    BX_TRY(BestXDirectRegWrite(handle,0x404508,sizeof(bx_int32),0));

    for (i=0;i<BX_RTBEH_MEMDEPTH;i++)
    {
      for (j=0;j<BX_RTBEH_MEMWIDTH;j++)
      {
        BX_TRY(BestXDirectRegRead(handle,0x404500,sizeof(bx_int16),&val));
        BESTX_FPRINTF(fp,"%04lx  ",val);
      }
      BESTX_FPRINTF(fp,"\n",val);
    }
    
    
    
    BX_TRY(BestXExerciserRunMode(handle,1));
  
  } /* TRY */

  BESTX_FCLOSE(fp);

  /* Append the host database */
  BestXPrintDB(handle,filename);

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * EXPORT bx_errtype BestXRunRam (
 *
 * Purpose  : run infinite loop in ram
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXRunRam (
  bx_handletype handle,
  bx_int32 jmpAddr,
  bx_bool immediate
  )
{
  BX_DECLARE_FUNCNAME("BestXRunRam []");
  BX_TRY_VARS_NO_PROG;

  bx_int8 in_zw[IN_RUN_RAM];
  bx_int8ptr bp;
  bx_int8 imm8;

  BX_TRY_BEGIN {
    BX_TRY (BestXHasFirmware(handle) == BX_TRUE ? BX_E_OK : BX_E_TBD);

    imm8 = (bx_int8) immediate;

    bp = BestXLong2Stream(in_zw, &jmpAddr, 1);  /* into the bytestream */
    bp = BestXByteCopy(bp, &imm8, 1);  /* into the bytestream */

    BX_TRY (BestXBasicCommand (handle,
             CMD_RUN_RAM,
             in_zw, IN_RUN_RAM,
             NULL, OUT_RUN_RAM));
  }

  BX_TRY_CATCH {
    /* enter error handling here */
  }

  return (BX_TRY_RET);

}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXMemAlloc (
 *
 * Purpose  : allocate buffer in card's memory
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXBoardMemAlloc (
  bx_handletype handle,
  bx_int32 size,
  bx_int32 * addr
  )
{
  BX_DECLARE_FUNCNAME("BestXBoardMemAlloc []");
  BX_TRY_VARS_NO_PROG;

  bx_int8 in_zw[IN_MEM_ALLOC];
  bx_int8 out_zw[OUT_MEM_ALLOC];
  bx_int16 outlen;
  bx_int8ptr bp;

  BX_TRY_BEGIN {
    BX_TRY (BestXHasFirmware(handle) == BX_TRUE ? BX_E_OK : BX_E_TBD);

    bp = BestXLong2Stream(in_zw, &size, 1);  /* into the bytestream */

    outlen = OUT_MEM_ALLOC;

    BX_TRY (BestXBasicCommand (handle,
             CMD_MEM_ALLOC,
             in_zw, IN_MEM_ALLOC,
             out_zw, &outlen));

    (void) BestXStream2Long(addr, out_zw, 1);
  }

  BX_TRY_CATCH {
    /* enter error handling here */
  }

  return (BX_TRY_RET);

}

/*---------------------------------------------------------------------------*
 * bx_errtype EXPORT BestXBoardMemFree (
 *
 * Purpose  : free buffer allocate in cards' memory
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXBoardMemFree (
  bx_handletype handle,
  bx_int32 addr
  )
{
  BX_DECLARE_FUNCNAME("BestXBoardMemFree []");
  BX_TRY_VARS_NO_PROG;

  bx_int8 in_zw[IN_MEM_FREE];
  bx_int8ptr bp;

  BX_TRY_BEGIN {
    BX_TRY (BestXHasFirmware(handle) == BX_TRUE ? BX_E_OK : BX_E_TBD);

    bp = BestXLong2Stream(in_zw, &addr, 1);  /* into the bytestream */

    BX_TRY (BestXBasicCommand (handle,
             CMD_MEM_FREE,
             in_zw, IN_MEM_FREE,
             NULL, OUT_MEM_FREE));
  }

  BX_TRY_CATCH {
    /* enter error handling here */
  }

  return (BX_TRY_RET);

}


/*---------------------------------------------------------------------------*
 * BestXStatusTestRead
 *
 * Purpose: Get status register for BX_STAT_TEST
 *---------------------------------------------------------------------------*/

static bx_errtype BestXStatusTestRead( 
  bx_handletype  handle,
  bx_int32 *status
)
{
  /* Registers from Mephisto */
  bx_int32 InfoReg=0;
  bx_int32 StatMonReg=0;
  bx_int32 RunStopStatReg=0;
  bx_int32 RunCtrlReg=0;
  bx_int32 AbortReg=0;

  /* Bits from Mephisto's Registers */
  bx_int32 run_stop_cmd_1;
  bx_int32 run_stop_status_2;
  bx_int32 all_tags_free_mask_3;
  bx_int32 all_tags_free_4;
  bx_int32 req_split_pend_5;
  bx_int32 ism_iabort_6;
  bx_int32 res_on_iabort_7;
  bx_int32 ism_tabort_8;
  bx_int32 res_on_tabort_9;
  bx_int32 cond_start_10;
  bx_int32 master_done_11;
  bx_int32 ism_stopped_12;
  bx_int32 mbreak_13;
  bx_int32 ex_sm_clear_14;

  BX_DECLARE_FUNCNAME("BestXStatusTestRead []");
  BX_TRY_VARS_NO_PROG;
  BX_FCT_PARAM_NULL_POINTER_CHK(status);

  BX_TRY_BEGIN
  {
    *status=0;

    /* Read relevant HW registers */
    BX_TRY(BestXDirectRegRead(handle,BX_REG_MSM_STAT_MON,sizeof(bx_int16),&StatMonReg));
    BX_TRY(BestXDirectRegRead(handle,BX_REG_MRUN_STOP_STATUS_REG,sizeof(bx_int16),&RunStopStatReg));
    BX_TRY(BestXDirectRegRead(handle,BX_REG_MSM_INFO_REG,sizeof(bx_int16),&InfoReg));
    BX_TRY(BestXDirectRegRead(handle,BX_REG_EX_RUN_CTRL_REG,sizeof(bx_int16),&RunCtrlReg));
    BX_TRY(BestXDirectRegRead(handle,BX_REG_MSM_ABORT_REG,sizeof(bx_int16),&AbortReg));

    run_stop_cmd_1       = RunCtrlReg     & 1<<0;
    run_stop_status_2    = RunStopStatReg & 1<<0;
    all_tags_free_mask_3 = RunStopStatReg & 1<<10;
    all_tags_free_4      = RunStopStatReg & 1<<2;
    req_split_pend_5     = RunStopStatReg & 1<<4;
    ism_iabort_6         = StatMonReg     & 1<<1;
    res_on_iabort_7      = AbortReg       & 1<<0;
    ism_tabort_8         = StatMonReg     & 1<<0;
    res_on_tabort_9      = AbortReg       & 1<<1;
    cond_start_10        = InfoReg        & 1<<5;
    master_done_11       = RunStopStatReg & 1<<3;
    ism_stopped_12       = InfoReg        & 1<<8;
    mbreak_13            = RunCtrlReg     & 1<<1;
    ex_sm_clear_14       = RunCtrlReg     & 1<<2;

    if (
        run_stop_cmd_1 && run_stop_status_2 && !(ism_iabort_6 && !res_on_iabort_7) && !cond_start_10 && !master_done_11 && !mbreak_13 && !ex_sm_clear_14 ||
        !run_stop_cmd_1 && !run_stop_status_2 && !all_tags_free_mask_3 && req_split_pend_5 && !mbreak_13 && !ex_sm_clear_14 ||
        !run_stop_cmd_1 && !run_stop_status_2 && !all_tags_free_mask_3 && !all_tags_free_4 && !req_split_pend_5 && !mbreak_13 && !ex_sm_clear_14
       )
    {
      *status |= BX_STAT_TEST_RUNNING;   
    }


    if (
        ex_sm_clear_14 ||
        ism_stopped_12 && mbreak_13 ||
        run_stop_cmd_1 && run_stop_status_2 && all_tags_free_4 && master_done_11 && !mbreak_13 && !ex_sm_clear_14 ||
        !run_stop_cmd_1 && !run_stop_status_2 && all_tags_free_mask_3 && !mbreak_13 && !ex_sm_clear_14 ||
        !run_stop_cmd_1 && !run_stop_status_2 && !all_tags_free_mask_3 && all_tags_free_4 && !mbreak_13 && !ex_sm_clear_14 ||
        !run_stop_cmd_1 && !run_stop_status_2 && !all_tags_free_mask_3 && !all_tags_free_4 && !req_split_pend_5  && !mbreak_13 && !ex_sm_clear_14 ||
        /* paused: */ ism_iabort_6 && !res_on_iabort_7 && !mbreak_13 && !ex_sm_clear_14
       )
    {
      *status |= BX_STAT_TEST_STOPPED;   
    }


    if (
         (!ism_iabort_6 || ism_iabort_6 && res_on_iabort_7) && !ism_tabort_8 && cond_start_10 && !mbreak_13 && !ex_sm_clear_14
       )
    {
      *status |= BX_STAT_TEST_WAITING;  
    }

    if (
         !run_stop_cmd_1 && !run_stop_status_2 && !all_tags_free_mask_3 && !all_tags_free_4 && !req_split_pend_5 && !mbreak_13 && !ex_sm_clear_14
       )
    {
      /* Special case:
         We are either running or stopped. 
         To find out we clear the ri_active flags, wait 10ms and read them out again.
      */

      bx_int32 SysStatMon=0;

      /* Clear ri_active bits */
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_SYS_STAT_MONITOR,sizeof(bx_int32),1<<15|1<<12));
      
      /* Wait 10ms */
      BESTX_SLEEP(10);
      
      /* Read ri_active bits */
      BX_TRY(BestXDirectRegRead(handle,BX_REG_SYS_STAT_MONITOR,sizeof(bx_int32),&SysStatMon));
      
    
      if (SysStatMon & 1<<15 /* || SysStatMon & 1<<12 */)
      {
        /* Still running (i.e. RI (bit 15) or CI (bit12) active) */
        *status |= BX_STAT_TEST_RUNNING;   
        *status &= ~(bx_int32)BX_STAT_TEST_STOPPED;
      }
      else
      {
        /* Already stopped */
        *status |= BX_STAT_TEST_STOPPED;   
        *status &= ~(bx_int32)BX_STAT_TEST_RUNNING;
      }
    }

    if ((*status & BX_STAT_TEST_WAITING) && (*status & BX_STAT_TEST_STOPPED))
    {
      /* Waiting bit has precedence before stopped bit (should be exclusive) */
      *status &= ~(bx_int32)BX_STAT_TEST_STOPPED;
    }

    if ((*status & BX_STAT_TEST_WAITING) && (*status & BX_STAT_TEST_STOPPED) ||
        (*status & BX_STAT_TEST_WAITING) && (*status & BX_STAT_TEST_RUNNING) ||
        (*status & BX_STAT_TEST_RUNNING) && (*status & BX_STAT_TEST_STOPPED)
       )
    {
      /* Should never happnen, because states are exclusive ! */
        *status |= BX_STAT_TEST_UNKNOWN;   
    }
    
    if (RunStopStatReg & 1<<3)
    {
      /* RI done bit */
       *status |= BX_STAT_TEST_DONE; 
    }

    if (RunStopStatReg & 1<<4)
    {
      /* RT waiting for outstanding completions */
       *status |= BX_STAT_TEST_COMPLETION;
    }


    if (StatMonReg & 1<<1)
    {
       /*Initiator abort occurred */
       *status |= BX_STAT_TEST_IABORT;   
    }

    if (StatMonReg & 1<<0)
    {
      /* Target abort occurred */
       *status |= BX_STAT_TEST_TABORT;
    }

    /*******************************/
    /* Not documented from here on */
    /*******************************/

    if (((RunStopStatReg & 1<<2)==0) && ((RunStopStatReg & 1<<10)==0))
    {
      /* Not all tags are free */
       *status |= BX_STAT_TEST_TAGBUSY; 
    }

    if ((RunStopStatReg & 1<<9)==0)
    {
      /* CI pipeline not empty */
       *status |= BX_STAT_TEST_CIPIPEFULL; 
    }

    if (RunCtrlReg & 1<<0)
    {
      /* RunStop */
       *status |= BX_STAT_TEST_RUNSTOP; 
    }

    if (RunCtrlReg & 1<<1)
    {
      /* Break state */
       *status |= BX_STAT_TEST_BREAK;
    }

    if (RunCtrlReg & 1<<2)
    {
      /* Clear */
       *status |= BX_STAT_TEST_CLEAR;
    }

    if ( (RunStopStatReg & 1<<3)                         /* DONE bit set       */                 || 
         (StatMonReg & 1<<1 && ((AbortReg & 1<<0)==0))   /* IABORT occured and stop on iabort */  ||
         (StatMonReg & 1<<0 && ((AbortReg & 1<<1)==0))   /* TABORT occured and stop on tabort */  
       )
    {
      /* Enhanced RI done bit */
       *status |= BX_STAT_TEST_INDONE;  
    }
  }

  BX_ERRETURN(BX_TRY_RET);

}

/*---------------------------------------------------------------------------*
 * BestXStatusCfgRead
 *
 * Purpose: Get status register for BX_STAT_CFGSTAT
 *          This is the emulated contents of the testcard status register
 *          at offset 0x52 (16bit) in config space, which is directly only
 *          accessible via port PCI, but not per DBI access.
 *---------------------------------------------------------------------------*/

static bx_errtype BestXStatusCfgRead( 
  bx_handletype  handle,
  bx_int32 *status
)
{
  bx_int32 val=0;

  BX_DECLARE_FUNCNAME("BestXStatusCfgRead []");
  BX_TRY_VARS_NO_PROG;
  BX_FCT_PARAM_NULL_POINTER_CHK(status);

  BX_TRY_BEGIN
  {
    *status=0;

    /* Bits 28-31 are fix '1111' */
    *status |= 0xf0000000;

    /* Bit 27: Triggered */
    BX_TRY(BestXStatusRead(handle,BX_STAT_TRC_TRIGGER,&val));
    *status |= (val?1:0)<<27; 

    /* Bit 26: Trace running */
    BX_TRY(BestXStatusRead(handle,BX_STAT_TRC_RUNNING,&val));
    *status |= (val?1:0)<<26; 

    /* Bit 25: Protocol error */
    BX_TRY(BestXStatusRead(handle,BX_STAT_OBS_ERR,&val));
    *status |= (val?1:0)<<25; 

    /* Bit 24: Split error received */
    BX_TRY(BestXStatusRead(handle,BX_STAT_SPLITFAIL,&val));
    *status |= (val?1:0)<<24; 

    /* Bit 23: Wether exerciser is in INIT mode */
    BX_TRY(BestXDirectRegRead(handle,BX_REG_EX_CLK_SWITCH_STAT_REG,sizeof(bx_int32),&val));
    *status |= (val?0:1)<<23; 

    /* Bit 22: Wether RI is running */
    BX_TRY(BestXDirectRegRead(handle,BX_REG_EX_RUN_CTRL_REG,sizeof(bx_int16),&val));
    *status |= (val&0x1?1:0)<<22; 

    /* Bit 21: Wether Completer pipeline is empty */
    BX_TRY(BestXDirectRegRead(handle,BX_REG_MRUN_STOP_STATUS_REG,sizeof(bx_int32),&val));
    *status |= (val&1<<9?1:0)<<21; 

    /* Bit 20: Wether all tags are free */
    BX_TRY(BestXDirectRegRead(handle,BX_REG_MRUN_STOP_STATUS_REG,sizeof(bx_int32),&val));
    *status |= (val&1<<2?1:0)<<21; 
    
    /* Bit 19: Initiator abort occured */
    BX_TRY(BestXStatusRead(handle,BX_STAT_IABORT,&val));
    *status |= (val?1:0)<<19; 
    
    /* Bit 18: SERR signaled */
    BX_TRY(BestXStatusRead(handle,BX_STAT_ERR_SERR,&val));
    *status |= (val?1:0)<<18; 
    
    /* Bit 17: PERR signaled */
    BX_TRY(BestXStatusRead(handle,BX_STAT_ERR_PERR,&val));
    *status |= (val?1:0)<<17; 
    
    /* Bit 16: Datacompare error */
    BX_TRY(BestXStatusRead(handle,BX_STAT_CMP_FAIL,&val));
    *status |= (val?1:0)<<16; 

    /* Bits 15-0 are covered by interrupt and mailbox status.
       We leave them zero here 
    */
  }

  BX_ERRETURN(BX_TRY_RET);

}




/********************************************************************
  Board properties **************************************************
 *******************************************************************/

/*---------------------------------------------------------------------------*
 * BestXBoardSet
 *
 * Purpose: Set board property on host-DB. No card access ! 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXBoardSet(
  bx_handletype handle,
  bx_boardtype prop,
  bx_int32 val
)
{
  BX_DECLARE_FUNCNAME("BestXBoardSet [boardset]");
  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_TRY(BestXParamCheck(handle, BX_PARAM_BOARD, (bx_int32)prop, val));

    BOARD(prop)=val;
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXBoardGet
 *
 * Purpose: Get board property from host-DB. No card access ! 
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXBoardGet(
    bx_handletype handle,
    bx_boardtype prop,
    bx_int32 * val
)
{
  BX_DECLARE_FUNCNAME("BestXBoardGet [boardget]");

  BX_TRY_VARS_NO_PROG;
  BX_FCT_PARAM_NULL_POINTER_CHK(val);

  BX_TRY_BEGIN
  {
    /* range checking */
    BX_TRY(BestXParamProbe(handle, BX_PARAM_BOARD, (bx_int32)prop));
    BX_TRY_FCT_PARAM_NULL_POINTER(val);
     
    /* get property from host DB */
    *val=(bx_int32)BOARD(prop);
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXBoardDefaultSet
 *
 * Purpose: Set board properties on host-DB to default. No card access ! 
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXBoardDefaultSet(
    bx_handletype handle
)
{
  BX_DECLARE_FUNCNAME("BestXBoardDefaultSet [boarddefset]");

  BX_TRY_VARS_NO_PROG;
  bx_int32 i;

  /* Generic info for range checking */
  const bx_generic_infotype *GenInfo;
  const bx_param_infotype *ParamInfo;

  BX_TRY_BEGIN
  {
    /* Get pointer to generic info */
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_BOARD, &GenInfo));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      BX_TRY(BestXParamInfoGet(handle, BX_PARAM_BOARD,
            i, &ParamInfo, (bx_int32)BX_INDEX_SEARCH));

      /* Set it to default */
      BX_TRY(BestXBoardSet(handle,
              ParamInfo->proptyp.boardprop,
              ParamInfo->defaultval));
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}


/*---------------------------------------------------------------------------*
 * BestXBoardRead
 *
 * Purpose: Reads board properties from card to host-DB. 
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXBoardRead(
    bx_handletype handle
)
{
  BX_DECLARE_FUNCNAME("BestXBoardRead [boardread]");
  BX_TRY_VARS_NO_PROG;

  bx_int32 i;

  /* Generic info for range checking */
  const bx_generic_infotype *GenInfo;
  const bx_param_infotype *ParamInfo;
  bx_int32 prop,val,reg;

  BX_TRY_BEGIN
  {
    /* Get pointer to generic info */
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_BOARD, &GenInfo));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      BX_TRY(BestXParamInfoGet(handle, BX_PARAM_BOARD,
            i, &ParamInfo, (bx_int32)BX_INDEX_SEARCH));

      /* We need an individual implementation for each property */
      switch (ParamInfo->proptyp.boardprop)
      {
        case BX_BOARD_DISPLAY:
          prop=BX_BOARD_DISPLAY;
          if(BestXHasFirmware(handle))
          {
            BX_TRY(BestXAbstractPropLongGet(handle, CMD_BOARD_PROP_GET,
                  (bx_int8) prop, &val));
          }
          else
          {
            val=BX_BOARD_DISPLAY_RESERVED;
          }
          BOARD(prop)=val; 
          break;
        case BX_BOARD_RESPECTBIOS:
        case BX_BOARD_BUSCLOCK:     
          /* CAPI-DB only properties; do not exist on card */
          break;

        case BX_BOARD_TRIGIO0_MODE:
          prop=BX_BOARD_TRIGIO0_MODE;
          BX_TRY(BestXDirectRegRead(handle,BX_REG_TRIG0_MODE_REG,sizeof(bx_int32),&val));
          BOARD(prop)=val;
          break;

        case BX_BOARD_TRIGIO1_MODE:
          prop=BX_BOARD_TRIGIO1_MODE;
          BX_TRY(BestXDirectRegRead(handle,BX_REG_TRIG1_MODE_REG,sizeof(bx_int32),&val));
          BOARD(prop)=val;
          break;
        case BX_BOARD_TRIGIO2_MODE:
          prop=BX_BOARD_TRIGIO2_MODE;
          BX_TRY(BestXDirectRegRead(handle,BX_REG_TRIG2_MODE_REG,sizeof(bx_int32),&val));
          BOARD(prop)=val;
          break;
          
        case BX_BOARD_TRIGIO3_MODE:
          prop=BX_BOARD_TRIGIO3_MODE;
          BX_TRY(BestXDirectRegRead(handle,BX_REG_TRIG3_MODE_REG,sizeof(bx_int32),&val));
          BOARD(prop)=val;
          break;

        case BX_BOARD_TRIGIO0_OUT:
        case BX_BOARD_TRIGIO1_OUT:
        case BX_BOARD_TRIGIO2_OUT:
        case BX_BOARD_TRIGIO3_OUT:
          prop=ParamInfo->proptyp.value;
          reg = BX_REG_TRIG0_SLCT_REG + 4UL * (prop - (bx_int32) BX_BOARD_TRIGIO0_OUT);
          BX_TRY(BestXDirectRegRead(handle,reg,sizeof(bx_int32),&val));
          BOARD(prop)= (val + (prop - (bx_int32) BX_BOARD_TRIGIO0_OUT)) % 4UL;
          break;
        case BX_BOARD_PCIXCAP:
          /* This property cannot be read directly from HW.
             Instead, we read a copy of the relay status from Flash */
          BX_TRY(BestXDirectRegRead(handle,BX_RELAY_STATUS,sizeof(bx_int16),&val));
          if (BestXIsOffline(handle))
          {
            val=0x4b; /* PCIX-66 */
          }
          
          /* If this assertion fails, the Flash is surely not yet programmed,
             i.e. BestXBoardProg() has not yet been executed before (production issue) */
          assert((val&0xf0)==(0xf0& ~(val<<4))); /* Check complement */
          prop=BX_BOARD_PCIXCAP;
          switch (val & 0x70) /* Check bits 4,5,6 (i.e. relays A,B,C) only */
          {
            case 1<<4:
              /* Relay A set */  
              BOARD(prop)=BX_BOARD_PCIXCAP_PCIX266;         
              break;
            case 1<<5:
              /* Relay B set */  
              BOARD(prop)=BX_BOARD_PCIXCAP_PCI;         
              break;
            case 1<<6:
              /* Relay C set */  
              BOARD(prop)=BX_BOARD_PCIXCAP_PCIX66;         
              break;
            case 0:
              /* Relays A,B,C unset */
              BOARD(prop)=BX_BOARD_PCIXCAP_PCIX133;         
              break;
            case 1<<4|1<<5:
              /* Relays A,B set */
              BOARD(prop)=BX_BOARD_PCIXCAP_PCIX533;         
              break;
            default:
              BX_E_ERROR_MSG_SET("BestXBoardRead: Invalid PCIXCAP value read from Flash");
              BX_TRY_ERROR(BX_E_ERROR);
              break;
          }
          break;
        case BX_BOARD_M66EN:
          /* This property cannot be read directly from HW.
             Instead, we read a copy of the relay status from Flash */
          BX_TRY(BestXDirectRegRead(handle,BX_RELAY_STATUS,sizeof(bx_int16),&val));
          if (BestXIsOffline(handle))
          {
            val=0x4b; /* PCIX-66 */
          }

          /* If this assertion fails, the Flash is surely not yet programmed,
             i.e. BestXBoardProg() has not yet been executed before (production issue) */
          assert((val&0xf0)==(0xf0& ~(val<<4))); /* Check complement */
          prop=BX_BOARD_M66EN;
          if (val & 1<<7)
          {
            /* Relay D set */
            BOARD(prop)=BX_BOARD_M66EN_GND;         
          }
          else
          {
            /* Relay D unset */
            BOARD(prop)=BX_BOARD_M66EN_OPEN;         
          }
          break;
        default:
          /* Unknown, not implemented property. */
          BX_TRY_FAIL(BX_E_INVALID_CASE);     /*Should never happen*/
          break;
      } /* switch */
    } /* for */
  } /* TRY_BEGIN */

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXBoardProg
 *
 * Purpose: Writes board properties from host DB to card to. 
 *---------------------------------------------------------------------------*/
bx_errtype EXPORT BestXBoardProg(
    bx_handletype handle
)
{
  BX_DECLARE_FUNCNAME("BestXBoardProg [boardprog]");
  BX_TRY_VARS_NO_PROG;

  bx_int32 i;

  /* Generic info for range checking */
  const bx_generic_infotype *GenInfo;
  const bx_param_infotype *ParamInfo;
  bx_int32 prop;
  bx_int32 val;
  bx_int16 relayval=0;
  bx_int32 reg;
  bx_int32 relaySectorAddress;
  bx_int8  relaySectorData[2];

  BX_TRY_BEGIN
  {
    /* Get pointer to generic info */
    BX_TRY(BestXGenInfoGet(handle, BX_PARAM_BOARD, &GenInfo));

    for (i = 0; i < GenInfo->num_elem; i++)
    {
      /* Get pointer to i-th (existing) property */
      BX_TRY(BestXParamInfoGet(handle, BX_PARAM_BOARD,
            i, &ParamInfo, (bx_int32)BX_INDEX_SEARCH));

      /* We need an individual implementation for each property */
      switch (ParamInfo->proptyp.boardprop)
      {
        case BX_BOARD_DISPLAY:
          prop=BX_BOARD_DISPLAY;
          if(BestXHasFirmware(handle))
          {
            BX_TRY(BestXAbstractPropLongSet(handle, CMD_BOARD_PROP_SET,
            (bx_int8) prop,(bx_int32)BOARD(prop)));
          }
          break;
        case BX_BOARD_RESPECTBIOS:
        case BX_BOARD_BUSCLOCK:     
          /* CAPI-DB only properties; do not exist on card */
          break;

        /* trigger out modes: (input, opendrain, totempole) */
        case BX_BOARD_TRIGIO0_MODE: /* utilize that props and registers */
        case BX_BOARD_TRIGIO1_MODE: /* are next to eachother, respectively! */
        case BX_BOARD_TRIGIO2_MODE:
        case BX_BOARD_TRIGIO3_MODE:
          prop=ParamInfo->proptyp.value;
          val=BOARD(prop);
          reg = BX_REG_TRIG0_MODE_REG + 4UL * (prop - (bx_int32) BX_BOARD_TRIGIO0_MODE);
          BX_TRY(BestXDirectRegWrite(handle, reg, 2UL, val));
          break;

        /* trigger out functions: (trig-gen, trace-trig, datacmp, proterr) */
        case BX_BOARD_TRIGIO0_OUT: /* NOTE: special case here. Different */
        case BX_BOARD_TRIGIO1_OUT: /*       trigger-outs have different */
        case BX_BOARD_TRIGIO2_OUT: /*       bit-assignments in HW! */
        case BX_BOARD_TRIGIO3_OUT: /*       the following algorithm takes */
          /* care of that! */
          prop=ParamInfo->proptyp.value;
          val = (4UL - (prop - (bx_int32) BX_BOARD_TRIGIO0_OUT) + BOARD(prop)) %  4UL;
          reg = BX_REG_TRIG0_SLCT_REG + 4UL * (prop - (bx_int32) BX_BOARD_TRIGIO0_OUT);
          BX_TRY(BestXDirectRegWrite(handle, reg, 2UL, val));
          break;
        case BX_BOARD_PCIXCAP:
          prop=ParamInfo->proptyp.value;

          switch (BOARD(prop))
          {
            case BX_BOARD_PCIXCAP_PCIX266:
              /* Set Relay A */  
              relayval|=1<<4; 
              break;
            case BX_BOARD_PCIXCAP_PCI:
              /* Set Relay B */  
              relayval|=1<<5; 
              break;
            case BX_BOARD_PCIXCAP_PCIX66:
              /* Set Relay C */  
              relayval|=1<<6; 
              break;
            case BX_BOARD_PCIXCAP_PCIX133:
              /* Relays A,B,C unset */
              break;
            case BX_BOARD_PCIXCAP_PCIX533:
              /* Set relays A,B */
              relayval|=1<<4|1<<5; 
              break;
            default:
              BX_E_ERROR_MSG_SET("BestXBoardProg: Invalid PCIXCAP value");
              BX_TRY_ERROR(BX_E_ERROR);
              break;
          }
          break;
        case BX_BOARD_M66EN: /* Relay D */
          prop=ParamInfo->proptyp.value;

          switch (BOARD(prop))
          {
            case BX_BOARD_M66EN_OPEN:
              /* Relay D unset */
              break;
            case BX_BOARD_M66EN_GND:
              /* Set relay D */
              relayval|=1<<7;
              break;
            default:
              BX_E_ERROR_MSG_SET("BestXBoardProg: Invalid PCIXCAP value");
              BX_TRY_ERROR(BX_E_ERROR);
              break;
          }
          break;
        default:
          /* Unknown, not implemented property. */
          BX_TRY_FAIL(BX_E_INVALID_CASE); /* Should never happen.  */
          break;
      } /* switch */
    } /* for */

    if (!BestXHasMephisto(handle))
    {
      /* Card has relays onboard and 
         properties PCIXCAP and M66EN have been mapped to variable relayval above.

         D C B A            Set relay (bits 3-0 are complement of bits 7-4)
         7 6 5 4 3 2 1 0    Bits of BX_REG_SYS_RELAY_REG
         X 0 1 0 X 1 0 1    PCI     (a5 (PCI33) resp 2d (PCI66)) (Relay B set)
         X 1 0 0 X 0 1 1    PCIX66  (4b) (Relay C set)
         X 0 0 0 X 1 1 1    PCIX133 (0f) (Relays A,B,C unset)
         X 0 0 1 X 1 1 0    PCIX266 (1e) (Relay A set)
         X 0 1 1 X 1 0 0    PCIX533 (3c) (Relays A,B set)

         1 X X X 0 X X X    M66EN_GND  (Relay D set)
         0 X X X 1 X X X    M66EN_OPEN (Relay D unset)
         
         Program relays:
         Bit 0-3 are relays A-D to be reset (bits not yet set),
         Bit 4-7 are relays A-D to be set   (bits already set above via PCIXCAP and M66EN properties)
       */
      relayval|=~(relayval>>4)&0xf; /* set bits 0-3 as complement of bits 4-7 */
      BX_TRY(BestXDirectRegWrite(handle, BX_REG_SYS_RELAY_REG, 2UL, (bx_int32)relayval)); /* programs relays A-D */
    
      /* Because we cannot read back the status of the relays directly and 
         because the PULoader should always be able to restore valid relay settings,
         we additionally store the relay status in flash memory, if needed.
      */
    
      BX_TRY(BestXDirectRegRead(handle, BX_RELAY_STATUS, 2UL,&val));
      if (((bx_int16)(val&0xff))!=relayval)
      {
        /* Values differ, so we need to reprogram flash */

        /* Erase flash for sector BX_RELAY_STATUS */
        relaySectorAddress=BX_RELAY_STATUS;
        BX_TRY(BestXEraseFlash (handle,1,&relaySectorAddress));

        /* Write flash for BX_RELAY_STATUS */
        BestXWord2Stream(relaySectorData,&relayval,1);      
        BX_TRY(BestXWriteFlash (handle,BX_RELAY_STATUS,relaySectorData,2));
      }
    }
  }
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * static BestXPCIFrequencyRead
 *
 * Purpose: Measures the Frequency of the PCI clock and returns the value
 *          as a DWord. The unit of the result is Hz.
 * CAUTION:
 *    This function is not re-entrant.
 *---------------------------------------------------------------------------*/

static bx_errtype BestXPCIFrequencyRead(
  bx_handletype handle,
  bx_int32  n_DBI,       /* <IN>  Time interval in DBI-clock cycles    */
  bx_int32  type,        /* <IN>  0=BUSSPEED, 1= PLLSPEED              */
  bx_int32 *f_PCI        /* <OUT> Measured PCI clock frequency in Hz   */
)
{
  BX_DECLARE_FUNCNAME("BestXPCIFrequencyRead []");

  bx_int32 f_DBI=BX_DBIFREQ;        /* DBI bus frequency               */
  bx_int32 n_PCI;         /* measured number of clock cycles */
  bx_int32 Status;        /* Status of measurement           */

  bx_int32 MaxTries=3000; /* Try three times until timeout             */
  bx_int32 TryNo;         /* Try counter counts from 1 to MaxTries     */

  bx_int8 in_zw;
  bx_int8 buf[4];
  bx_int16 length;

  BX_TRY_VARS_NO_PROG;

  BX_FCT_PARAM_NULL_POINTER_CHK(f_PCI);

  *f_PCI=0;

  BX_TRY_BEGIN
  {
    switch (bx_handlearray[handle].port) 
    {
#ifdef CUSTOM_OEM1
      case BX_PORT_OEM:
#endif
      case BX_PORT_PCI_CONF:
      case BX_PORT_PCI_IO:
      case BX_PORT_PRODUCTION:
        MaxTries=3000;
        break;
      default:
        MaxTries=3;
    }
  
    if (BestXIsOffline(handle))
    {
      /* 133 MHz in Offline mode */
      *f_PCI=133000000;
      BX_ERRETURN(BX_E_OK);
    }

    if (n_DBI==0)
    {
      /* Special case */
      BX_ERRETURN(BX_E_OK);
    }

    if (BestXHasFirmware(handle))
    {
      /* 29er: We cannot start measurement here in CAPI, because it interferes 
         with the onboard measurement for the frequency display */
      in_zw=(bx_int8) type;
      length=OUT_SYSTEM_INFO_GET;
      BX_TRY(BestXBasicCommand(handle, CMD_SYSTEM_INFO_GET, &in_zw, IN_SYSTEM_INFO_GET,buf,&length));
      (void) BestXStream2Long(f_PCI,buf,1);
    }
    else
    {
      /* no firmware, so no interfering frequency measurement onboard */
  
      /*  Measurement initialization */
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_LOCAL_SW_RESET,sizeof(bx_int16),0));   

      /* Write the measurement interval into the 
         decrementor preload register.
      */
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_FREQ_PL_REG,sizeof(bx_int32),n_DBI));   

      /* Start measurement by asserting the preload strobe.
         This copies the preload value into the decrementor counter. 
      */
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_FREQ_DECR_PL,sizeof(bx_int16),1));   
    
      /* Poll status until measure complete or timeout */
      TryNo = 0;
      do
      { 
        TryNo++;
        /* Read Status info */
        BX_TRY(BestXDirectRegRead(handle,BX_REG_FREQ_RDY_REG,sizeof(bx_int16),&Status)); 
        Status &= (type?0x19:0x7);
      }
      while (((Status) != 0) && (TryNo < MaxTries));

      if (Status == 0) /* Measurement successful */
      {
        /* Get measured PCI clocks */
        BX_TRY(BestXDirectRegRead(handle,
              (type?BX_REG_PLLFREQ_INCR32_REG:BX_REG_PCIFREQ_INCR32_REG),
                sizeof(bx_int32),&n_PCI));   
               
        /* Measurement Time interval im sec = n_DBI/f_DBI = n_PCI/f_PCI
           Formula for computing the PCI frequency then is:
           f_PCI = (f_DBI * n_PCI) / n_DBI
           Possible problem:
           The numerator may exceed MAXINT.
           This happens whenever f_DBI * nPCI >= 0xffffffff
           (i.e. PCI frequency >= MAXINT/n_DBI).
           For this, we divide the numerator and denominator by 2
           until the numerator is less equal MAXINT.
           if this is not possible, we round down.
        */
      
        /* Make sure there's no overflow in the numerator */
        while ( (0xFFFFFFFF / f_DBI) <= n_PCI )
        {
          if ( (f_DBI & 1) == 0)
          {
        f_DBI >>= 1;
          }
          else
          {
            if ( (n_PCI & 1) == 0 || (n_PCI > f_DBI) )
            {
              n_PCI >>=1;
            }
            else
            {
          f_DBI >>= 1;
            }
          }
          n_DBI >>= 1;
        } /* while */

        /* Compute the PCI bus frequency */
        *f_PCI = n_PCI * f_DBI;
        *f_PCI = *f_PCI / n_DBI;
      }
      else
      {
        /* measurement not successful */
        *f_PCI = 0;
      }
    } /* no firmware */
  }
  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXBoardInit
 *
 * Purpose: Switches card into run-mode.
 *          Inits handle->CardIsInProgMode.
 *          Clears force retry bit.
 *          
 *          Preconditions:
 *          Dynamic capabilities already present.
 *          DB initialized.
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXBoardInit(bx_handletype handle,bx_int32 SoftForce)
{
  BX_DECLARE_FUNCNAME("BestXBoardInit [boardinit]");

  bx_int32 RunMode,ForceRetryBit,val;

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* The board property BX_BOARD_BUSCLOCK is needed for switching between run and prog-mode,
       but is not yet initialized here (will be read from card later on).
       So we pre-set them here on the host. This only works, if the dynamic 
       capabilities and the host-DB are already initialized !
       If parameter force equals zero, this function will return an error, 
       if there is no or a slow PCI clock (-> use BestXOpenAdv(force=1)). 
    */
    
    /* Avoid a hard force */
    BX_TRY(BestXBoardDefaultSet(handle));
    BX_TRY(BestXBoardSet(handle,BX_BOARD_BUSCLOCK,BX_BOARD_BUSCLOCK_DEF));
 
    /* Lets look, wether exerciser is in Run- or Prog-Mode */
    BX_TRY(BestXDirectRegRead(handle,BX_REG_EX_CLK_SWITCH_STAT_REG,sizeof(bx_int16),&RunMode));
    bx_handlearray[handle].CardIsInProgrammingMode=(RunMode?0:1);
    if (RunMode==0)
    {
      /* We are in Programming mode, maybe the PU loader has been disabled per 
         DIP switch. Switch exerciser to runmode:
      */
       
      BX_TRY(BestXExerciserRunMode(handle,SoftForce));

      /* 
         If parameter force equals zero, this function will return an error, 
         if there is no or a slow PCI clock (-> use BestXOpenAdv(SoftForce=1)). 
      */
    }
    
    /* Lets look, wether analyzer is in Run- or Prog-Mode */
    BX_TRY(BestXDirectRegRead(handle,BX_REG_TR_CLK_SWITCH_STAT_REG,sizeof(bx_int16),&RunMode));
    if (RunMode==0)
    {
      /* We are in Programming mode. Switch to runmode.
      */
      BX_TRY(BestXAnalyzerRunMode(handle,SoftForce));
    }

    /* Lets look, wether performance is in Run- or Prog-Mode */
    BX_TRY(BestXDirectRegRead(handle,BX_REG_AN_CLK_SWITCH_STAT_REG,sizeof(bx_int16),&RunMode));
    if (RunMode==0)
    {
      /* We are in Programming mode. Switch to runmode.
      */
      BX_TRY(BestXPerfRunMode(handle,SoftForce));
    }

    /* After power up the target force retry bit is set */

    /* Clear the force retry bit, if not already cleared */
    BX_TRY(BestXDirectRegRead(handle,BX_REG_FORCE_RETRY_SYNC_BIT_REG,sizeof(bx_int16),&val));
    ForceRetryBit=(val&1?1:0);

    if (ForceRetryBit==1)
    { 
      /* Clear force retry bit */
      BX_TRY(BestXSyncRegWrite(handle,
                           
                             BX_REG_FORCE_RETRY_BIT_REG, /* Status register */
                             sizeof(bx_int16),
                             0x2,  /* Mask for Bit 1    */
                             0x2,  /* 1 in Bit 1=Ready  */
                           
                             BX_REG_FORCE_RETRY_BIT_REG, /* Control register */
                             sizeof(bx_int16),
                             0x1, /* Mask for bit 0 */
                             0,   /* 1=Retry        */
                           
                             BX_REG_FORCE_RETRY_SYNC_BIT_REG,  /* Sync register */
  
                             BX_REG_FORCE_RETRY_SYNC_BIT_REG, /* Force register */
                             sizeof(bx_int16),
                             SoftForce
                             ));
    }


    BX_TRY(EnableInterrupts(handle));

  } /* TRY */

  BX_ERRETURN(BX_TRY_RET);
}

/********************************************************************
  Expansion ROM *****************************************************
 *******************************************************************/

bx_errtype EXPORT BestXExpRomWrite(                         
  bx_handletype handle,
  bx_int32 numbytes,   
  bx_int8 *data 
)
{
  BX_DECLARE_FUNCNAME("BestXExpRomWrite []");

  bx_char reason[256];

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* Check numbytes */
    BESTX_SPRINTF(reason, "greater than expansion ROM (%lu bytes)",0x10000);
    BX_TRY_FCT_PARAM_R(1, numbytes > 0x10000, reason);
    BX_TRY_FAIL(numbytes & 0x01UL ? BX_E_NO_WORD_BOUNDARY : BX_E_OK);

    /* Check data */
    BX_TRY_FCT_PARAM_NULL_POINTER(data);
    
    BX_TRY(BestXCoreSwitch(handle));
    BX_TRY(BestXFlashProgPrepare(handle, BX_FSEC_EXPROM));
    BX_TRY(BestXHexImageLoad(handle,data,numbytes));
    BX_TRY(BestXHexImageProg(handle));
    BX_TRY(BestXBiosSwitch(handle,0 /* no card reset, just switch back to BIOS */));
  } 

  BX_ERRETURN(BX_TRY_RET);
}
  
bx_errtype EXPORT BestXExpRomRead(
  bx_handletype handle,
  bx_int32     offset,           
  bx_int32     numbytes, 
  bx_int8 *data  
)
{
  BX_DECLARE_FUNCNAME("BestXExpRomRead []");

  bx_char reason[256];

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* Check offset */
    BESTX_SPRINTF(reason, "greater than expansion ROM (%lu bytes)",0x10000);
    BX_TRY_FCT_PARAM_R(1, numbytes > 0x10000, reason);
    BX_TRY_FAIL(offset & 0x01UL ? BX_E_NO_WORD_BOUNDARY : BX_E_OK);

    /* Check numbytes */
    BESTX_SPRINTF(reason, "exceeds expansion ROM size (%lu bytes)",0x10000);
    BX_TRY_FCT_PARAM_R(2, offset+numbytes > 0x10000, reason);
    BX_TRY_FAIL(numbytes & 0x01UL ? BX_E_NO_WORD_BOUNDARY : BX_E_OK);

    /* Check data */
    BX_TRY_FCT_PARAM_NULL_POINTER(data);

    /* Read from card */
    BX_TRY(BestXDirectRegBlockRead(handle,0xc0000+offset,sizeof(bx_int16),0,data,sizeof(bx_int8),numbytes));
  } 

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * EnableInterrupts
 *
 * Purpose: Enables the interrupts
 *---------------------------------------------------------------------------*/

static bx_errtype EnableInterrupts(bx_handletype handle)
{
  BX_DECLARE_FUNCNAME("EnableInterrupts []");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* PCI Interrupts A,B,C,D */
    /* First set the clock count registers to a value > 0 */
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_INTA_CLK_CNT_REG,sizeof(bx_int16),1));
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_INTB_CLK_CNT_REG,sizeof(bx_int16),1));
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_INTC_CLK_CNT_REG,sizeof(bx_int16),1));
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_INTD_CLK_CNT_REG,sizeof(bx_int16),1));

    /* Second, enable the four interrupts */
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_INT_EN_REG,sizeof(bx_int16),0xf));
  } 

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * AbortClear
 *
 * Purpose: Clears the abort state
 *---------------------------------------------------------------------------*/

static bx_errtype AbortClear(bx_handletype handle)
{
  /* Get the state machine out of the abort state */

  bx_int32 AbortReg,StatusReg,RunControlReg,val,AlreadyBroken;

  BX_DECLARE_FUNCNAME("AbortClear []");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* Rememer BX_RIGEN_IABORT and BX_RIGEN_TABORT */
    BX_TRY(BestXDirectRegRead(handle,BX_REG_MSM_ABORT_REG,sizeof(bx_int16),&AbortReg));

    /* First try to clear */
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_MSM_STAT_MON,sizeof(bx_int16),0x0));
    BX_TRY(BestXDirectRegRead(handle,BX_REG_MSM_STAT_MON,sizeof(bx_int16),&StatusReg));

    if (StatusReg)
    {
      /* Clear was not successful, still in abort state.
 
         This happens when BX_RIGEN_IABORT resp. BX_RIGEN_TABORT is '0' 
         (i.e. stop execution, when abort has occurred; do not continue).
         This is the default.

         To make the clear successful, we temporary change BX_RIGEN_IABORT 
         resp. BX_RIGEN_TABORT to '1' (i.e. continue) and then retry the clear again.
       
         Problem: We dont want the RI to continue after getting out of the abort state
                  (HW does so automatically). So we have to execute a Break() before and
                  clear the break afterwards (if we are not already in the break state).
      */

      /* Check, wether we are already in Break-state */
      BX_TRY(BestXDirectRegRead(handle,BX_REG_EX_RUN_CTRL_SYNC_REG,sizeof(bx_int16),&RunControlReg));
    
      AlreadyBroken=(RunControlReg&1<<2?1:0);

      if (!AlreadyBroken)
      {
        /* RI would continue to execute as soon as we would clear the abort state.
           To avoid this we break it before 
        */
        BX_TRY(BestXExerciserBreak(handle));
      }

      /* Set props to 'SKIP' (i.e. continue) */
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_MSM_ABORT_REG,sizeof(bx_int16),0x3));

      /* Clear now should work: */
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_MSM_STAT_MON,sizeof(bx_int16),0x0));
      BX_TRY(BestXDirectRegRead(handle,BX_REG_MSM_STAT_MON,sizeof(bx_int16),&StatusReg));

      if (StatusReg)
      {
        /* We should never get here */
        BX_E_ERROR_MSG_SET("BestXStatusClear: Cannot clear abort state");
        BX_TRY_ERROR(BX_E_ERROR);
      }

      /* Set props to their previous value */
      BX_TRY(BestXDirectRegWrite(handle,BX_REG_MSM_ABORT_REG,sizeof(bx_int16),AbortReg));
    
      if (!AlreadyBroken)
      {
        /* Undo/Remove the previous Break() */
        BX_TRY(BestXDirectRegRead(handle,BX_REG_EX_RUN_CTRL_REG,sizeof(bx_int16),&val));
        BX_TRY(BestXDirectRegWrite(handle,BX_REG_EX_RUN_CTRL_REG,sizeof(bx_int16),val&0x5));
      }
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}


/*---------------------------------------------------------------------------*
 * BestXRequestAssert
 *
 * Purpose: 
 *  1. Asserts REQ#
 *  2. Waits for GNT# being asserted and the bus becoming idle
 *  3. Waits for the given number of clock cycles
 *  4. Deasserts REQ#
 *---------------------------------------------------------------------------*/

bx_errtype EXPORT BestXRequestAssert(
  bx_handletype    handle,
  bx_int32 clocks
)
{
  BX_DECLARE_FUNCNAME("BestXRequestAssert");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    BX_TRY(BestXDirectRegWrite(handle,BX_REG_BOARD_REQASSERT_REG,sizeof(bx_int32),clocks));
  }

  BX_ERRETURN(BX_TRY_RET);
}

/*---------------------------------------------------------------------------*
 * BestXWaitForDiagXfer
 *
 * Purpose: 
 *  Waits for (previously initiated) SMB bus transfer to finish
 *---------------------------------------------------------------------------*/

static bx_errtype BestXWaitForDiagXfer(handle)
{
  bx_int32 Try=10;
  bx_int32 readval=0;

  BX_DECLARE_FUNCNAME("BestXWaitForDiagXfer");

  BX_TRY_VARS_NO_PROG;

  BX_TRY_BEGIN
  {
    /* Read control reg. and wait for 06h, meaning transfer success (read bit remains set!) */
    while (Try>0)
    {
      BX_TRY(BestXDirectRegRead(handle,BX_REG_DIAG_CONTROL,sizeof(bx_int16),&readval));
      if (readval==0x6)
      {
        /* Success */           
        break;
      }
      Try--;
    }
    if (Try==0)
    {
      BX_E_ERROR_MSG_SET("BestXStatusRead: Timeout during read of a BX_STAT_DIAG value");
      BX_TRY_ERROR(BX_E_ERROR);
    }
  }

  BX_ERRETURN(BX_TRY_RET);
}
